//DRADIQUE Arduino Due edition with Phase Distortion synthesis, woo!

//COPYRIGHT 2018 J. van Kuik, Evil Turtle Productions

/*
2 12-bit DAC outputs, buffered into a stereo output
let's have a single mono core with stereo FX such as reverb and chorus
this will save processing

Core can be 8-note polyphonic initially, but maybe we can push it up to 16
it would be nice to layer 2 sounds like the Casio CZ series can.

Let's do 8 CZ-styel waveforms initially:
saw, square, pulse, double sine, half sine, reso saw, reso triangle, reso square

but we can add many more waveforms, heck even could do wavetables since there
is so much memory available on the Due... What to do with all this RAM?

==================================================================================

CASIO CZ SERIES INFO:
- DCO
8 or 16 digital oscillators, could be layered and thus halving the polyphony

- DCW ("filter")
The DCW of an oscillator is the magnitude of distortion that is applied to the reading angle of that oscillator's selected waveform. 
The DCW can be modified over time using an attack-decay-sustain-release (ADSR) envelope, thus changing the timbre of the sound over time.

- DCA
The DCA (which determined how loud a given oscillator was at a given moment) was also modulated 
by another dedicated 8-stage envelope generator. 
The DCW and DCA also had a "key follow" feature; which determined how much higher notes affected a sound, 
making the DCW have a more dull sound with less harmonics with higher notes, and making the DCA envelope faster for higher notes. 

- ENV
were eight stage envelope generators where each stage had a rate and level value. 
The rate value determined how fast the envelope would move; the level value 
would determine what pitch/filter cutoff/volume the envelope would have. There was a single sustain stage, and an end stage. 

- LFO
The LFO can use triangle, square, upwards (ramp) sawtooth, or downwards sawtooth waveforms. 
The modulatory effects of the LFO are controlled by three settings: speed, depth, and delay. 

- FX
It was possible to modulate the two voices in a two-voice patch in two different ways. 
Ring modulation had the output of one of the oscillators affect the volume of the other oscillator, 
resulting in a controlled distortion. 
Noise modulation caused the second voice in a two-voice patch to sound like digital noise, 
roughly simulating the effect of an analog synthesizer's noise source.

*/

/*

TODO:
- vibrato would be really cool
- need some detuning in dual mode (can be button 4)
- sequencer

BUGS:
- envelope release values change when switching params?

 */

#include <DueFlashStorage.h>
#include <efc.h>
#include <flash_efc.h>

DueFlashStorage dueFlashStorage;

#include "TABLES.h"
#include "DCO.h"

//Port manipulation inline faffery
inline void digitalWriteDirect(int pin, boolean val)
{
  if(val) g_APinDescription[pin].pPort -> PIO_SODR = g_APinDescription[pin].ulPin;
  else    g_APinDescription[pin].pPort -> PIO_CODR = g_APinDescription[pin].ulPin;
}

inline int digitalReadDirect(int pin)
{
  return !!(g_APinDescription[pin].pPort -> PIO_PDSR & g_APinDescription[pin].ulPin);
}

//hardware defines
#define KEYS1 42
#define KEYS2 44
#define KEYS3 46
#define KEYS4 48
#define KEYS5 50
#define KEYS6 52

#define KEYSA 37
#define KEYSB 39
#define KEYSC 41
#define KEYSD 43
#define KEYSE 45
#define KEYSF 47
#define KEYSG 49
#define KEYSH 51
#define KEYSI 53

//front panel buttons
#define MODESW_1 8
#define MODESW_2 9
#define MODESW_3 10

#define BUTT_TEMPO_UP  0
#define BUTT_TEMPO_DWN 1
#define BUTT_PATT_UP   2
#define BUTT_PATT_DWN  3
#define BUTT_CHORMODE  4
#define BUTT_CHOR_ONOFF 5

#define BUTT_STORE     16
#define BUTT_LOAD      17

#define BUTT_EDIT 24
#define BUTT_1    25
#define BUTT_2    26
#define BUTT_3    27
#define BUTT_4    28
#define BUTT_5    29
#define BUTT_6    41
#define BUTT_7    40
#define BUTT_8    39
#define BUTT_9    38
#define BUTT_0    37
#define BUTT_ENTER 36

#define USR_KNOB1 A0
#define USR_KNOB2 A1

#define USR_BEEP_PIN 4

#define OSCS_AMOUNT 10
DCO oscs[OSCS_AMOUNT];

//encoder
DCO_param* enc1_var;
DCO_param* enc2_var;

DCO_param* enc1_var_temp;
DCO_param* enc2_var_temp;

uint16_t tempknobA;
uint16_t tempknobB;

//user interface
//pressing edit button again while in a mode cancels it

//edit -> select param set number -> user knobs change 2 param values + enter

//optionally:
//edit -> rotate user knob + enter -> select param number + enter

//modes
//0 = nothing
//1 = select param

byte mode = 0;
uint16_t buttidprev = 255;
bool selectingwaveform = false;

bool buttons[42] = 
{
  false
};

//synthmode 0 = poly
//synthmode 1 = duo
//synthmode 2 = something :)
//synthmode 3 = something else :)
int synthmode = 0; 

uint16_t timercounter = 0;
volatile uint16_t timercounter_int = 0;
bool timerstep = false;

uint16_t keyblocks[9] =
{
  41, 51, 47, 43, 39, 37, 53, 49, 45
};

uint16_t keysections[6] =
{
  42, 44, 46, 48, 50, 52
};

bool keys[49] = 
{
  false, false, false, false, false, false, false, false, false, false, false, false, 
  false, false, false, false, false, false, false, false, false, false, false, false, 
  false, false, false, false, false, false, false, false, false, false, false, false, 
  false, false, false, false, false, false, false, false, false, false, false, false, 
  false
};

//chorus stuff
#define CHOR_MAX_D 350
#define CHOR_MID_D 250
#define CHOR_MIN_D 200
bool chor_active = false;
volatile uint16_t chor_max = CHOR_MAX_D;
volatile int32_t chor_bufferL[CHOR_MAX_D+250];
volatile int32_t chor_bufferR[CHOR_MAX_D+250];
volatile uint16_t chor_counter = 0;
volatile uint16_t chor_depth = 300;
volatile bool chor_count_up = true;
volatile uint16_t chor_counterR = 0;
volatile uint16_t chor_depthR = 100;
volatile bool chor_count_upR = true;

//user preset storage
byte stor_currindex = 255;
byte stor_editvalue = 0;
byte stor_editoffset = 1;

//main setup
void setup()
{
  //global params default values
  //dummy
  dummy.value = 0;
  dummy.mini = 0;
  dummy.maxi = 16;
  dummy.expo = false;

  //params
  detuneA.value = 1024;       //512 - 2048
  detuneA.mini = 512;
  detuneA.maxi = 2048;
  detuneA.expo = false;
  detuneB.value = 1024;       //512 - 2048
  detuneB.mini = 512;
  detuneB.maxi = 2048;
  detuneB.expo = false;
  
  env_attack.value = 256;     //1-512
  env_attack.mini = 1;
  env_attack.maxi = 512;
  env_attack.expo = true;
  env_decay.value = 512;      //1-512
  env_decay.mini = 1;
  env_decay.maxi = 512;
  env_decay.expo = true;
  env_sustain.value = 1023;    //0-1023
  env_sustain.mini = 0;
  env_sustain.maxi = 1023;
  env_sustain.expo = true;
  env_release.value = 256;    //1-512
  env_release.mini = 1;
  env_release.maxi = 512;
  env_release.expo = true;
  
  mod_attack.value = 512;     //1-512
  mod_attack.mini = 1;
  mod_attack.maxi = 512;
  mod_attack.expo = true;
  mod_decay.value = 512;      //1-512
  mod_decay.mini = 1; 
  mod_decay.maxi = 512; 
  mod_decay.expo = true;
  mod_sustain.value = 1023;    //0-1023
  mod_sustain.mini = 0;
  mod_sustain.maxi = 1023;
  mod_sustain.expo = true;
  mod_release.value = 512;    //1-512
  mod_release.mini = 1;
  mod_release.maxi = 512;
  mod_release.expo = true;
  
  mod_cutoff.value = 0;     //0-31
  mod_cutoff.mini = 0; 
  mod_cutoff.maxi = 31;
  mod_cutoff.expo = true;
  mod_resonance.value = 0; //0-16
  mod_resonance.mini = 0; 
  mod_resonance.maxi = 16;
  mod_resonance.expo = true;

  waveformA_id.value;
  waveformA_id.mini = 0;
  waveformA_id.maxi = TAB_WAVEFORMS_NR-1;
  waveformA_id.expo = false;
  waveformB_id.value;
  waveformB_id.mini = 0;
  waveformB_id.maxi = TAB_WAVEFORMS_NR-1;
  waveformB_id.expo = false;
    
  modulatorA_id.value = 0;
  modulatorA_id.mini = 0;
  modulatorA_id.maxi = TAB_MODULATORS_NR-1;
  modulatorA_id.expo = false;
  modulatorB_id.value = 0;
  modulatorB_id.mini = 0;
  modulatorB_id.maxi = TAB_MODULATORS_NR-1;
  modulatorB_id.expo = false;
  
  //encoder defaults
  enc1_var = &dummy;
  enc2_var = &dummy;  

  enc1_var_temp = &dummy;
  enc2_var_temp = &dummy;
  
  //synth stuff
  for(int j = 0; j < OSCS_AMOUNT; j++)
  {
    oscs[j].note = 255; //255 = no note
    oscs[j].gate = false;
    oscs[j].env_stage = false;
    oscs[j].mod_stage = false;
    oscs[j].busy = false;

    oscs[j].envelope = 0;
    oscs[j].modulation = 0;
    
    oscs[j].freq = 0;
    oscs[j].counter = 0;
    oscs[j].resocounter = 0;
    oscs[j].mod_depth = 0;
    oscs[j].env_val = 0;
    
    //array append to combine two waveforms CZ style
    for(int i = 0; i < 512; i++)
    {
      oscs[j].waveform[i] = TAB_WAVEFORMS[waveformA_id.value][i];
      oscs[j].waveform[i+512] = TAB_WAVEFORMS[waveformB_id.value][i];
      
      oscs[j].modulator[i] = TAB_MODULATORS[modulatorA_id.value][i];
      oscs[j].modulator[i+512] = TAB_MODULATORS[modulatorB_id.value][i];
    }
  }
  
  
  //IO and timers
  analogWriteResolution(12); //12-bit DAC (4096)
  analogReadResolution(10); //10-bit ADC (1024)
  
  analogWrite(DAC0,0);  // Enables DAC0
  analogWrite(DAC1,0);  // Enables DAC0

  pinMode(MODESW_1, INPUT_PULLUP);
  pinMode(MODESW_2, INPUT_PULLUP);
  pinMode(MODESW_3, INPUT_PULLUP);

  pinMode(KEYS1, OUTPUT);
  pinMode(KEYS2, OUTPUT);
  pinMode(KEYS3, OUTPUT);
  pinMode(KEYS4, OUTPUT);
  pinMode(KEYS5, OUTPUT);
  pinMode(KEYS6, OUTPUT);

  pinMode(KEYSA, INPUT);
  pinMode(KEYSB, INPUT);
  pinMode(KEYSC, INPUT);
  pinMode(KEYSD, INPUT);
  pinMode(KEYSE, INPUT);
  pinMode(KEYSF, INPUT);
  pinMode(KEYSG, INPUT);
  pinMode(KEYSH, INPUT);
  pinMode(KEYSI, INPUT);

  pinMode(27, OUTPUT);
  pinMode(29, OUTPUT);
  pinMode(31, OUTPUT);
  pinMode(33, OUTPUT);

  pinMode(26, INPUT);
  pinMode(28, INPUT);
  pinMode(30, INPUT);
  pinMode(32, INPUT);
  pinMode(34, INPUT);
  pinMode(36, INPUT);

  analogRead(USR_KNOB1);
  analogRead(USR_KNOB2);

  pinMode(USR_BEEP_PIN, OUTPUT);

  //for testing
  Serial.begin(9600);
  Serial.print("Initializing\n");


  //TIMER
  pmc_set_writeprotect(false);
  pmc_enable_periph_clk(ID_TC4);
 
  //we want wavesel 01 with RC 
  TC_Configure(TC1, 1, TC_CMR_WAVE | TC_CMR_WAVSEL_UP_RC | TC_CMR_TCCLKS_TIMER_CLOCK2);
  TC_SetRC(TC1, 1, 238); // sets <> 44.1 Khz interrupt rate
 
  TC_Start(TC1, 1);
 
  // enable timer interrupts on the timer
  TC1->TC_CHANNEL[1].TC_IER=TC_IER_CPCS;
  TC1->TC_CHANNEL[1].TC_IDR=~TC_IER_CPCS;
 
  //Enable the interrupt in the nested vector interrupt controller 
  //TC4_IRQn where 4 is the timer number * timer channels (3) + the channel 
  //number (=(1*3)+1) for timer1 channel1 
  NVIC_EnableIRQ(TC4_IRQn);
}


//MAIN LOOP ********************************
void loop()
{
  if(timercounter > 110 && !timerstep) //run at ~5ms
  {
    timercounter = 0;
    timerstep = true;

    //process envelopes
    for(int i = 0; i < OSCS_AMOUNT; i++)
    {
      if(oscs[i].busy)
      {
        if(oscs[i].gate) //attack, decay (stage), sustain phase
        {
          //modulation
          if(!oscs[i].mod_stage) //attack
          {
            oscs[i].modulation += mod_attack.value;
            if(oscs[i].modulation > 1023)
            {
              oscs[i].modulation = 1023;
              oscs[i].mod_stage = true;
            }
          }
          else //decay + sustain
          {
            if(oscs[i].modulation > mod_sustain.value)
              oscs[i].modulation -= mod_decay.value;
            else
              oscs[i].modulation = mod_sustain.value;
          }

          //envelope
          if(!oscs[i].env_stage) //attack
          {
            oscs[i].envelope += env_attack.value;
            if(oscs[i].envelope > 1023)
            {
              oscs[i].envelope = 1023;
              oscs[i].env_stage = true;
            }
          }
          else //decay + sustain
          {
            if(oscs[i].envelope > env_sustain.value)
              oscs[i].envelope -= env_decay.value;
            else
              oscs[i].envelope = env_sustain.value;
          }
        }
        else //release phase
        {
          //modulation
          if(oscs[i].modulation > mod_release.value)
            oscs[i].modulation -= mod_release.value;
          else
            oscs[i].modulation = 0;

          //amplitude
          if(oscs[i].envelope > env_release.value)
            oscs[i].envelope -= env_release.value;
          else
          {
            oscs[i].busy = false;
            oscs[i].envelope = 0;
            oscs[i].modulation = 0;
            oscs[i].resofreq = 0;
            oscs[i].freq = 0;
            oscs[i].counter = 0;
            oscs[i].resocounter = 0;
          }
        }

        //modulation goes to 128 now, ~72 seems to be CZ max range
        //so that should be pretty good
        oscs[i].resofreq = (oscs[i].modulation * mod_cutoff.value) >> 4;
        oscs[i].mod_depth = oscs[i].resofreq >> 4;
      }
    }
  }

  if(timercounter > 110 && timerstep) //~5ms
  {
    timercounter = 0;
    timerstep = false;
    
    //front panel
    //SCAN TWO USER KNOBS =================================================
    /*if((ADC->ADC_ISR & 0xC0) == 0xC0 && !mode) //sample available and we are not editing
    {
      if(enc1_var->expo)
      {
        enc1_var->value = map(USR_EXPCONV[analogRead(USR_KNOB1)], 0, 1023, enc1_var->mini, enc1_var->maxi);
        enc2_var->value = map(USR_EXPCONV[analogRead(USR_KNOB2)], 0, 1023, enc2_var->mini, enc2_var->maxi);
      }
      else
      {
        enc1_var->value = enc1_var->maxi + enc1_var->mini - map(analogRead(USR_KNOB1), 0, 1023, enc1_var->mini, enc1_var->maxi);
        enc2_var->value = enc2_var->maxi + enc2_var->mini - map(analogRead(USR_KNOB2), 0, 1023, enc2_var->mini, enc2_var->maxi);
      }
    }*/

    //SCAN SYNTH MODE SWITCH
    //synth 0 = 10
    //synth 1 = nothing
    //synth 2 = 8
    //synth 3 = 9

    if(digitalReadDirect(MODESW_3) == LOW && synthmode != 0)
    {
      synthmode = 0;
      USR_BEEP(false);
    }
    else if(digitalReadDirect(MODESW_1) == LOW && synthmode != 2)
    {
      synthmode = 2;
      USR_BEEP(false);
    }
    else if(digitalReadDirect(MODESW_2) == LOW && synthmode != 3)
    {
      synthmode = 3;
      USR_BEEP(false);
    }
    else if(digitalReadDirect(MODESW_1) &&
            digitalReadDirect(MODESW_2) &&
            digitalReadDirect(MODESW_3) && synthmode != 1)
    {
      synthmode = 1;
      USR_BEEP(false);
    }

    //INTERFACE BUTTONS
    uint16_t x = 0;
    uint16_t y = 0;
    uint16_t buttid = 255;

    digitalWriteDirect(27, HIGH);
    for(int i = 26; i < 37; i+=2)
    {
      uint16_t buttnr = x+y;
      if(digitalReadDirect(i)==HIGH)
      {
        if(!buttons[buttnr])
        {
          buttons[buttnr] = true;
          buttid = buttnr;
        }
      }
      else if(buttons[buttnr])
      {
        buttons[x+y] = false;
      }
      x++;
    }
    digitalWriteDirect(27, LOW);

    y+=6;
    digitalWriteDirect(29, HIGH);
    for(int i = 26; i < 37; i+=2)
    {
      uint16_t buttnr = x+y;
      if(digitalReadDirect(i)==HIGH)
      {
        if(!buttons[buttnr])
        {
          buttons[buttnr] = true;
          buttid = buttnr;
        }
      }
      else if(buttons[buttnr])
      {
        buttons[x+y] = false;
      }
      x++;
    }
    digitalWriteDirect(29, LOW);

    y+=6;
    digitalWriteDirect(31, HIGH);
    for(int i = 26; i < 37; i+=2)
    {
      uint16_t buttnr = x+y;
      if(digitalReadDirect(i)==HIGH)
      {
        if(!buttons[buttnr])
        {
          buttons[buttnr] = true;
          buttid = buttnr;
        }
      }
      else if(buttons[buttnr])
      {
        buttons[x+y] = false;
      }
      x++;
    }
    digitalWriteDirect(31, LOW);

    y+=6;
    digitalWriteDirect(33, HIGH);
    for(int i = 26; i < 37; i+=2)
    {
      uint16_t buttnr = x+y;
      if(digitalReadDirect(i)==HIGH)
      {
        if(!buttons[buttnr])
        {
          buttons[buttnr] = true;
          buttid = buttnr;
        }
      }
      else if(buttons[buttnr])
      {
        buttons[x+y] = false;
      }
      x++;
    }
    digitalWriteDirect(33, LOW);

    //do button actions
    //chorus ****************************************************
    if(buttid == BUTT_CHOR_ONOFF)
    {
      chor_active = !chor_active;
      USR_BEEP(false);
    }
    else if(buttid == BUTT_CHORMODE)
    {
      if(chor_max == CHOR_MAX_D)
        chor_max = CHOR_MID_D;
      else
        chor_max = CHOR_MAX_D;

      USR_BEEP(false);
    }

    //user patch storage ****************************************************
    if(buttid == BUTT_STORE && mode == 0) //store patch
    {
      stor_editvalue = 0;
      stor_editoffset = 1;
      mode = 2;
      USR_BEEP(false);
    }
    else if(buttid == BUTT_STORE && mode == 2) //cancel
    {
      stor_editvalue = 0;
      stor_editoffset = 1;
      mode = 0;     
      USR_BEEP(true);
    }

    if(buttid == BUTT_LOAD && mode == 0) //load patch
    {
      stor_editvalue = 0;
      stor_editoffset = 1;
      mode = 4;     
      USR_BEEP(false);
    }
    else if(buttid == BUTT_LOAD && mode == 4) //cancel
    {
      stor_editvalue = 0;
      stor_editoffset = 1;
      mode = 0;     
      USR_BEEP(true);
    }

    if(buttid == BUTT_ENTER && mode == 2 && stor_editvalue != 0) //preset store
    {
      USR_BEEP(false);
      mode = 3;

      if(stor_editvalue < 100)
      {
        //stop interrupts
        noInterrupts();
        
        stor_currindex = stor_editvalue * 27; //27 byte total values are stored, so this is offset

        //env
        dueFlashStorage.write(stor_currindex, lowByte(env_attack.value)); //write first byte of value
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, highByte(env_attack.value)); //write second byte of value
        stor_currindex++; //increment index
        
        dueFlashStorage.write(stor_currindex, lowByte(env_decay.value)); //write first byte of value
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, highByte(env_decay.value)); //write second byte of value
        stor_currindex++; //increment index

        dueFlashStorage.write(stor_currindex, lowByte(env_sustain.value)); //write first byte of value
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, highByte(env_sustain.value)); //write second byte of value
        stor_currindex++; //increment index

        dueFlashStorage.write(stor_currindex, lowByte(env_release.value)); //write first byte of value
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, highByte(env_release.value)); //write second byte of value
        stor_currindex++; //increment index

        //mod env
        dueFlashStorage.write(stor_currindex, lowByte(mod_attack.value)); //write first byte of value
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, highByte(mod_attack.value)); //write second byte of value
        stor_currindex++; //increment index
        
        dueFlashStorage.write(stor_currindex, lowByte(mod_decay.value)); //write first byte of value
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, highByte(mod_decay.value)); //write second byte of value
        stor_currindex++; //increment index

        dueFlashStorage.write(stor_currindex, lowByte(mod_sustain.value)); //write first byte of value
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, highByte(mod_sustain.value)); //write second byte of value
        stor_currindex++; //increment index

        dueFlashStorage.write(stor_currindex, lowByte(mod_release.value)); //write first byte of value
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, highByte(mod_release.value)); //write second byte of value
        stor_currindex++; //increment index

        //filter
        dueFlashStorage.write(stor_currindex, lowByte(mod_cutoff.value)); //write first byte of value
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, highByte(mod_cutoff.value)); //write second byte of value
        stor_currindex++; //increment index

        dueFlashStorage.write(stor_currindex, lowByte(mod_resonance.value)); //write first byte of value
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, highByte(mod_resonance.value)); //write second byte of value
        stor_currindex++; //increment index

        //waveform ID's, can just be bytes because there's only a few values
        dueFlashStorage.write(stor_currindex, (byte)waveformA_id.value);
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, (byte)waveformB_id.value);
        stor_currindex++; //increment index

        //modulator waveform ID's
        dueFlashStorage.write(stor_currindex, (byte)modulatorA_id.value);
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, (byte)modulatorB_id.value);
        stor_currindex++; //increment index


        //chorus, can be 1 byte for on/off bool and 2 bytes for delay_max        
        dueFlashStorage.write(stor_currindex, (byte)chor_active);
        stor_currindex++; //increment index
        
        dueFlashStorage.write(stor_currindex, lowByte(chor_max)); //write first byte of value
        stor_currindex++; //increment index
        dueFlashStorage.write(stor_currindex, highByte(chor_max)); //write second byte of value
        stor_currindex++; //increment index

        //that's all flosdfkhdsfhsdofihdsf
        interrupts();
      } 

      stor_currindex = 255;
      stor_editvalue = 0;
      stor_editoffset = 1;
      mode = 0;
    }
    else if(buttid == BUTT_ENTER && mode == 4 && stor_editvalue != 0) //preset load
    {
      USR_BEEP(false);
      mode = 5;     
      if(stor_editvalue < 100)
      {
        noInterrupts();

        stor_currindex = stor_editvalue * 27;
        
        //envelope        
        byte lobyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        byte hibyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        env_attack.value = ((uint16_t)hibyte << 8) + lobyte;

        lobyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        hibyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        env_decay.value = ((uint16_t)hibyte << 8) + lobyte;

        lobyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        hibyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        env_sustain.value = ((uint16_t)hibyte << 8) + lobyte;

        lobyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        hibyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        env_release.value = ((uint16_t)hibyte << 8) + lobyte;


        //mod env
        lobyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        hibyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        mod_attack.value = ((uint16_t)hibyte << 8) + lobyte;

        lobyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        hibyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        mod_decay.value = ((uint16_t)hibyte << 8) + lobyte;

        lobyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        hibyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        mod_sustain.value = ((uint16_t)hibyte << 8) + lobyte;

        lobyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        hibyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        mod_release.value = ((uint16_t)hibyte << 8) + lobyte;

        //filter
        lobyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        hibyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        mod_cutoff.value = ((uint16_t)hibyte << 8) + lobyte;

        lobyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        hibyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        mod_resonance.value = ((uint16_t)hibyte << 8) + lobyte;

        //waveform ID's (single bytes!)
        waveformA_id.value = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        waveformB_id.value = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        //modulator ID's (single bytes!)
        modulatorA_id.value = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        modulatorB_id.value = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        for(int j = 0; j < OSCS_AMOUNT; j++)
        {        
          //array append to combine two waveforms CZ style
          for(int i = 0; i < 512; i++)
          {
            oscs[j].waveform[i] = TAB_WAVEFORMS[waveformA_id.value][i];
            oscs[j].waveform[i+512] = TAB_WAVEFORMS[waveformB_id.value][i];
            
            oscs[j].modulator[i] = TAB_MODULATORS[modulatorA_id.value][i];
            oscs[j].modulator[i+512] = TAB_MODULATORS[modulatorB_id.value][i];
          }
        }

        //chorus
        chor_active = (bool)dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        lobyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;
        hibyte = dueFlashStorage.read(stor_currindex);
        stor_currindex++;

        chor_max = ((uint16_t)hibyte << 8) + lobyte;
        
        //is all, now fuck off
        interrupts();
      }

      stor_currindex = 255;
      stor_editvalue = 0;
      stor_editoffset = 1;
      mode = 0;
    }

    if(mode == 2 || mode == 4) //scan buttons if we are in save or load mode
    {
      switch(buttid)
      {
        case BUTT_1:
          USR_BEEP(false);
          stor_editvalue *= stor_editoffset;
          stor_editoffset = 10;
          stor_editvalue += 1;
        break;

        case BUTT_2:
          USR_BEEP(false);
          stor_editvalue *= stor_editoffset;
          stor_editoffset = 10;
          stor_editvalue += 2;
        break;

        case BUTT_3:
          USR_BEEP(false);
          stor_editvalue *= stor_editoffset;
          stor_editoffset = 10;
          stor_editvalue += 3;
        break;

        case BUTT_4:
          USR_BEEP(false);
          stor_editvalue *= stor_editoffset;
          stor_editoffset = 10;
          stor_editvalue += 4;
        break;

        case BUTT_5:
          USR_BEEP(false);
          stor_editvalue *= stor_editoffset;
          stor_editoffset = 10;
          stor_editvalue += 5;
        break;

        case BUTT_6:
          USR_BEEP(false);
          stor_editvalue *= stor_editoffset;
          stor_editoffset = 10;
          stor_editvalue += 6;
        break;

        case BUTT_7:
          USR_BEEP(false);
          stor_editvalue *= stor_editoffset;
          stor_editoffset = 10;
          stor_editvalue += 7;
        break;

        case BUTT_8:
          USR_BEEP(false);
          stor_editvalue *= stor_editoffset;
          stor_editoffset = 10;
          stor_editvalue += 8;
        break;

        case BUTT_9:
          USR_BEEP(false);
          stor_editvalue *= stor_editoffset;
          stor_editoffset = 10;
          stor_editvalue += 9;
        break;

        case BUTT_0:
          USR_BEEP(false);
          stor_editvalue *= stor_editoffset;
          stor_editoffset = 10;
        break;

        default:
        break;
      }
    }

    //param editing **************************************************** 
    if(buttid == BUTT_EDIT && mode == 0) //go into edit mode
    {
      mode = 1;

      //temp store param values so they can be reset when cancelled      
      temp_env_attack = env_attack.value;
      temp_env_decay = env_decay.value;
      temp_env_sustain = env_sustain.value;
      temp_env_release = env_release.value;
      
      temp_mod_attack = mod_attack.value;
      temp_mod_decay = mod_decay.value;
      temp_mod_sustain = mod_sustain.value;
      temp_mod_release = mod_release.value;
      
      temp_mod_cutoff = mod_cutoff.value;
      temp_mod_resonance = mod_resonance.value;

      temp_waveformA_id = waveformA_id.value;
      temp_waveformB_id = waveformB_id.value;

      temp_modulatorA_id = modulatorA_id.value;
      temp_modulatorB_id = modulatorB_id.value;

      USR_BEEP(false);
    }
    else if(buttid == BUTT_EDIT && mode == 1) //cancel edit mode
    {
      mode = 0;

      //load in all old values and the waveforms      
      env_attack.value = temp_env_attack;
      env_decay.value = temp_env_decay;
      env_sustain.value = temp_env_sustain;
      env_release.value = temp_env_release;
      
      mod_attack.value = temp_mod_attack;
      mod_decay.value = temp_mod_decay;
      mod_sustain.value = temp_mod_sustain;
      mod_release.value = temp_mod_release;
      
      mod_cutoff.value = temp_mod_cutoff;
      mod_resonance.value = temp_mod_resonance;

      waveformA_id.value = temp_waveformA_id;
      waveformB_id.value = temp_waveformB_id;

      modulatorA_id.value = temp_modulatorA_id;
      modulatorB_id.value = temp_modulatorB_id;

      for(int j = 0; j < OSCS_AMOUNT; j++)
      {        
        //array append to combine two waveforms CZ style
        for(int i = 0; i < 512; i++)
        {
          oscs[j].waveform[i] = TAB_WAVEFORMS[waveformA_id.value][i];
          oscs[j].waveform[i+512] = TAB_WAVEFORMS[waveformB_id.value][i];
          
          oscs[j].modulator[i] = TAB_MODULATORS[modulatorA_id.value][i];
          oscs[j].modulator[i+512] = TAB_MODULATORS[modulatorB_id.value][i];
        }
      }

      //assign dummies again
      enc1_var = &dummy;
      enc2_var = &dummy;

      USR_BEEP(true);
    }

    if(buttid == BUTT_ENTER && mode == 1) //pressed enter in edit mode
    {
      //this means that changed values are permanent
      mode = 0;
      
      //assign dummies again
      enc1_var = &dummy;
      enc2_var = &dummy;

      USR_BEEP(false);
    }

    //PROCESS EDITING FAFFERY
    if(mode == 1)
    {
      //number buttons select param pair
      //only when rotating a knob does param change, to prevent instant changes when
      //switching param pairs

      //scan the user's knob
      uint16_t tempA;
      uint16_t tempB;
      if((ADC->ADC_ISR & 0xC0) == 0xC0) //sample available and we are not editing
      {
        if(enc1_var->expo)
        {
          tempA = map(USR_EXPCONV[analogRead(USR_KNOB1)], 0, 1023, enc1_var->mini, enc1_var->maxi);
          tempB = map(USR_EXPCONV[analogRead(USR_KNOB2)], 0, 1023, enc2_var->mini, enc2_var->maxi);
        }
        else
        {
          tempA = enc1_var->maxi + enc1_var->mini - map(analogRead(USR_KNOB1), 0, 1023, enc1_var->mini, enc1_var->maxi);
          tempB = enc2_var->maxi + enc2_var->mini - map(analogRead(USR_KNOB2), 0, 1023, enc2_var->mini, enc2_var->maxi);
        }
      }
      
      switch(buttid)
      {
        case BUTT_1:
          USR_BEEP(false);
          enc1_var = &waveformA_id;
          enc2_var = &waveformB_id;
          selectingwaveform = true;
          tempknobA = tempA;
          tempknobB = tempB;
        break;

        case BUTT_2:
          USR_BEEP(false);
          enc1_var = &modulatorA_id;
          enc2_var = &modulatorB_id;
          selectingwaveform = true;
          tempknobA = tempA;
          tempknobB = tempB;
        break;

        case BUTT_3:
          USR_BEEP(false);
          enc1_var = &mod_cutoff;
          enc2_var = &mod_resonance;
          selectingwaveform = false;
          tempknobA = tempA;
          tempknobB = tempB;
        break;

        case BUTT_4:
          USR_BEEP(false);
          enc1_var = &detuneA;
          enc2_var = &detuneB;
          selectingwaveform = false;
          tempknobA = tempA;
          tempknobB = tempB;
        break;

        case BUTT_5:
          USR_BEEP(false);
          enc1_var = &mod_attack;
          enc2_var = &env_attack;
          selectingwaveform = false;
          tempknobA = tempA;
          tempknobB = tempB;
        break;

        case BUTT_6:
          USR_BEEP(false);
          enc1_var = &mod_decay;
          enc2_var = &env_decay;
          selectingwaveform = false;
          tempknobA = tempA;
          tempknobB = tempB;
        break;

        case BUTT_7:
          USR_BEEP(false);
          enc1_var = &mod_sustain;
          enc2_var = &env_sustain;
          selectingwaveform = false;
          tempknobA = tempA;
          tempknobB = tempB;
        break;

        case BUTT_8:
          USR_BEEP(false);
          enc1_var = &mod_release;
          enc2_var = &env_release;
          selectingwaveform = false;
          tempknobA = tempA;
          tempknobB = tempB;
        break;

        case BUTT_9:
          USR_BEEP(false);
          //enc1_var = &modulatorA_id; //this could be vibrato depth and speed
          //enc2_var = &modulatorB_id;
          selectingwaveform = false;
          tempknobA = tempA;
          tempknobB = tempB;
        break;

        case BUTT_0:
          USR_BEEP(false);
          //enc1_var = &modulatorA_id; //no idea what this'll be
          //enc2_var = &modulatorB_id;
          selectingwaveform = false;
          tempknobA = tempA;
          tempknobB = tempB;
        break;

        default:
        break;
      }
      
      if(tempA != tempknobA)
      {
        tempknobA = tempA;
        enc1_var->value = tempA;
        if(selectingwaveform)
        {
          //we have selected a waveform so change things
          for(int j = 0; j < OSCS_AMOUNT; j++)
          {        
            //array append to combine two waveforms CZ style
            for(int i = 0; i < 512; i++)
            {
              oscs[j].waveform[i] = TAB_WAVEFORMS[waveformA_id.value][i];
              oscs[j].waveform[i+512] = TAB_WAVEFORMS[waveformB_id.value][i];
              
              oscs[j].modulator[i] = TAB_MODULATORS[modulatorA_id.value][i];
              oscs[j].modulator[i+512] = TAB_MODULATORS[modulatorB_id.value][i];
            }
          }
        }
      }
      
      if(tempB != tempknobB)
      {
        tempknobB = tempB;
        enc2_var->value = tempB;
        if(selectingwaveform)
        {
          //we have selected a waveform so change things
          for(int j = 0; j < OSCS_AMOUNT; j++)
          {        
            //array append to combine two waveforms CZ style
            for(int i = 0; i < 512; i++)
            {
              oscs[j].waveform[i] = TAB_WAVEFORMS[waveformA_id.value][i];
              oscs[j].waveform[i+512] = TAB_WAVEFORMS[waveformB_id.value][i];
              
              oscs[j].modulator[i] = TAB_MODULATORS[modulatorA_id.value][i];
              oscs[j].modulator[i+512] = TAB_MODULATORS[modulatorB_id.value][i];
            }
          }
        }
      }
      
    }

    buttidprev = buttid;
  }

  //keyboard scanning, as fast as possible:
  uint16_t keyid = 0;

  for(int x = 0; x < 6; x++)
  {
    digitalWriteDirect(keysections[x], HIGH);

    for(int y = 0; y < 8; y++)
    {
      if(digitalReadDirect(keyblocks[y]) == HIGH)
      {
        keyid = (y*6)+x;

        if(keys[keyid] == false)
        {
          keys[keyid] = true;

          if(synthmode != 1) //poly and speshiul durrrrr
          {
            bool alltaken = true;
            for(int i = 0; i < OSCS_AMOUNT; i++)
            {              
              if(!oscs[i].busy) //not busy
              {
                oscs[i].busy = true;
                oscs[i].gate = true;
                oscs[i].note = keyid;
                oscs[i].freq = NOTES[48-keyid];
                alltaken = false;
                break;
              }
            }
            
            if(alltaken) //nothing available, time to be a dirty thief
            {
              for(int i = 0; i < OSCS_AMOUNT; i++)
              {
                if(!oscs[i].gate) //in release phase, we do a steal owo
                {
                  //oscs[i].busy = true; //not needed here
                  oscs[i].gate = true;
                  oscs[i].note = keyid;
                  oscs[i].freq = NOTES[48-keyid];
                  break;
                }
              }
            }
          }
          else //if synthmode is 1 = dual
          {
            bool alltaken = true;
            for(int i = 0; i < OSCS_AMOUNT; i+=2)
            {
              if(!oscs[i].busy)
              {
                oscs[i].busy = true;
                oscs[i].gate = true;
                oscs[i].note = keyid;
                oscs[i].freq = (NOTES[48-keyid] * detuneA.value) >> 10;
                oscs[i+1].busy = true;
                oscs[i+1].gate = true;
                oscs[i+1].note = keyid;
                oscs[i+1].freq = (NOTES[48-keyid] * detuneB.value) >> 10;
                alltaken = false;
                break;
              }
            }

            if(alltaken) //nothing available, time to be a dirty thief
            {
              for(int i = 0; i < OSCS_AMOUNT; i+=2)
              {
                if(!oscs[i].gate) //in release phase, we do a steal owo
                {
                  //oscs[i].busy = true;
                  oscs[i].gate = true;
                  oscs[i].note = keyid;
                  oscs[i].freq = (NOTES[48-keyid] * detuneA.value) >> 10;
                  //oscs[i+1].busy = true;
                  oscs[i+1].gate = true;
                  oscs[i+1].note = keyid;
                  oscs[i+1].freq = (NOTES[48-keyid] * detuneB.value) >> 10;
                  break;
                }
              }
            }
          }
        }
      }
      else
      {
        keyid = (y*6)+x;
        if(keys[keyid] == true)
        {
          keys[keyid] = false;

          if(synthmode != 1) //poly and speshiul durrrrr
          {
            for(int i = 0; i < OSCS_AMOUNT; i++)
            {
              if(oscs[i].note == keyid)
              {
                oscs[i].gate = false;
                oscs[i].mod_stage = false;
                oscs[i].env_stage = false;
                oscs[i].note = 255;
                break;
              }
            }
          }
          else //dual
          {
            for(int i = 0; i < OSCS_AMOUNT; i+=2)
            {
              if(oscs[i].note == keyid)
              {
                oscs[i].gate = false;
                oscs[i].mod_stage = false;
                oscs[i].env_stage = false;
                oscs[i].note = 255;
                oscs[i+1].gate = false;
                oscs[i+1].mod_stage = false;
                oscs[i+1].env_stage = false;
                oscs[i+1].note = 255;
                break;
              }
            }
          }
        }
      }
    }

    /*if(x == 0)
    {
      if(digitalRead(keyblocks[9]) == HIGH)
      {
        keyid = 48;
        Serial.print("keyid = ");
        Serial.print(keyid);
        Serial.print("\n");
      }
    }*/

    digitalWriteDirect(keysections[x], LOW);
  }  
}


//user interface feedback (a bit crude)
void USR_BEEP(bool hi)
{
  bool beeper = false;
  int dtime = 500;
  if(hi)
    dtime = 250;

  int doscs = 30;
  if(hi)
    doscs = 60;
  for(int i = 0; i < doscs; i++)
  {
    if(beeper)
    {
      beeper = false;
      digitalWriteDirect(USR_BEEP_PIN, HIGH);
    }
    else
    {
      beeper = true;
      digitalWriteDirect(USR_BEEP_PIN, LOW);
    }
      
    delayMicroseconds(dtime); //1000Hz
  }

  digitalWriteDirect(USR_BEEP_PIN, LOW);
}


//Interrupt *********************************
void TC4_Handler()
{
  //reset timer and output value ===============================================
  TC_GetStatus(TC1, 1);
  int32_t outvalL = 0; //32-bit for more headroom
  int32_t outvalR = 0; //starting at the midpoint

  //process DCO's ==============================================================
  //DCO phase distortion and output
  oscs[0].counter += oscs[0].freq;
  uint16_t index = oscs[0].counter + (oscs[0].modulator[oscs[0].counter>>6] * oscs[0].mod_depth);
  int32_t result = oscs[0].waveform[index>>6];

  //update envelope when the amplitude is smallest
  //decided to do this in the crossover over zero, since certain waveforms, like the square wave,
  //kind only jump from min to max and thus break the updating
  //this will also sync the update with the phase, thus reducing the wierdness maybe?
  if(oscs[0].result_prev < 0 && result > 0)
    oscs[0].env_val = oscs[0].envelope;
  oscs[0].result_prev = result;
  outvalL = result * oscs[0].env_val; //moved this to here, as to reduce envelope influence on itself
  
  //resonance output, only for the second waveform, CZ style :)
  if(oscs[0].counter < 32768)
    oscs[0].resocounter = 0;
  else
  {
    oscs[0].resocounter += oscs[0].resofreq;
    outvalL += (tab_sine[oscs[0].resocounter>>7] * (65535-oscs[0].counter)>>9) * mod_resonance.value;
  }

  //7 more DCO's
  oscs[1].counter += oscs[1].freq;
  index = oscs[1].counter + (oscs[1].modulator[oscs[1].counter>>6] * oscs[1].mod_depth);
  result = oscs[1].waveform[index>>6];
  //env
  if(oscs[1].result_prev < 0 && result > 0)
    oscs[1].env_val = oscs[1].envelope;
  oscs[1].result_prev = result;
  outvalL += result * oscs[1].env_val;
  //reso
  if(oscs[1].counter < 32768)
    oscs[1].resocounter = 0;
  else
  {
    oscs[1].resocounter += oscs[1].resofreq;
    outvalL += (tab_sine[oscs[1].resocounter>>7] * (65535-oscs[1].counter)>>9) * mod_resonance.value;
  }

  //dco
  oscs[2].counter += oscs[2].freq;
  index = oscs[2].counter + (oscs[2].modulator[oscs[2].counter>>6] * oscs[2].mod_depth);
  result = oscs[2].waveform[index>>6];
  //env
  if(oscs[2].result_prev < 0 && result > 0)
    oscs[2].env_val = oscs[2].envelope;
  oscs[2].result_prev = result;
  outvalL += result * oscs[2].env_val;
  //reso
  if(oscs[2].counter < 32768)
    oscs[2].resocounter = 0;
  else
  {
    oscs[2].resocounter += oscs[2].resofreq;
    outvalL += (tab_sine[oscs[2].resocounter>>7] * (65535-oscs[2].counter)>>9) * mod_resonance.value;
  }

  //DCO
  oscs[3].counter += oscs[3].freq;
  index = oscs[3].counter + (oscs[3].modulator[oscs[3].counter>>6] * oscs[3].mod_depth);
  result = oscs[3].waveform[index>>6];
  //env
  if(oscs[3].result_prev < 0 && result > 0)
    oscs[3].env_val = oscs[3].envelope;
  oscs[3].result_prev = result;
  outvalL += result * oscs[3].env_val;
  //reso
  if(oscs[3].counter < 32768)
    oscs[3].resocounter = 0;
  else
  {
    oscs[3].resocounter += oscs[3].resofreq;
    outvalL += (tab_sine[oscs[3].resocounter>>7] * (65535-oscs[3].counter)>>9) * mod_resonance.value;
  }

  //DCO
  oscs[4].counter += oscs[4].freq;
  index = oscs[4].counter + (oscs[4].modulator[oscs[4].counter>>6] * oscs[4].mod_depth);
  result = oscs[4].waveform[index>>6];
  //env
  if(oscs[4].result_prev < 0 && result > 0)
    oscs[4].env_val = oscs[4].envelope;
  oscs[4].result_prev = result;
  outvalL += result * oscs[4].env_val;
  //reso
  if(oscs[4].counter < 32768)
    oscs[4].resocounter = 0;
  else
  {
    oscs[4].resocounter += oscs[4].resofreq;
    outvalL += (tab_sine[oscs[4].resocounter>>7] * (65535-oscs[4].counter)>>9) * mod_resonance.value;
  }

  //DCO
  oscs[5].counter += oscs[5].freq;
  index = oscs[5].counter + (oscs[5].modulator[oscs[5].counter>>6] * oscs[5].mod_depth);
  result = oscs[5].waveform[index>>6];
  //env
  if(oscs[5].result_prev < 0 && result > 0)
    oscs[5].env_val = oscs[5].envelope;
  oscs[5].result_prev = result;
  outvalL += result * oscs[5].env_val;
  //reso
  if(oscs[5].counter < 32768)
    oscs[5].resocounter = 0;
  else
  {
    oscs[5].resocounter += oscs[5].resofreq;
    outvalL += (tab_sine[oscs[5].resocounter>>7] * (65535-oscs[5].counter)>>9) * mod_resonance.value;
  }

  //DCO
  oscs[6].counter += oscs[6].freq;
  index = oscs[6].counter + (oscs[6].modulator[oscs[6].counter>>6] * oscs[6].mod_depth);
  result = oscs[6].waveform[index>>6];
  //env
  if(oscs[6].result_prev < 0 && result > 0)
    oscs[6].env_val = oscs[6].envelope;
  oscs[6].result_prev = result;
  outvalL += result * oscs[6].env_val;
  //reso
  if(oscs[6].counter < 32768)
    oscs[6].resocounter = 0;
  else
  {
    oscs[6].resocounter += oscs[6].resofreq;
    outvalL += (tab_sine[oscs[6].resocounter>>7] * (65535-oscs[6].counter)>>9) * mod_resonance.value;
  }

  //DCO
  oscs[7].counter += oscs[7].freq;
  index = oscs[7].counter + (oscs[7].modulator[oscs[7].counter>>6] * oscs[7].mod_depth);
  result = oscs[7].waveform[index>>6];
  //env
  if(oscs[7].result_prev < 0 && result > 0)
    oscs[7].env_val = oscs[7].envelope;
  oscs[7].result_prev = result;
  outvalL += result * oscs[7].env_val;
  //reso
  if(oscs[7].counter < 32768)
    oscs[7].resocounter = 0;
  else
  {
    oscs[7].resocounter += oscs[7].resofreq;
    outvalL += (tab_sine[oscs[7].resocounter>>7] * (65535-oscs[7].counter)>>9) * mod_resonance.value;
  }

  //DCO
  oscs[8].counter += oscs[8].freq;
  index = oscs[8].counter + (oscs[8].modulator[oscs[8].counter>>6] * oscs[8].mod_depth);
  result = oscs[8].waveform[index>>6];
  //env
  if(oscs[8].result_prev < 0 && result > 0)
    oscs[8].env_val = oscs[8].envelope;
  oscs[8].result_prev = result;
  outvalL += result * oscs[8].env_val;
  //reso
  if(oscs[8].counter < 32768)
    oscs[8].resocounter = 0;
  else
  {
    oscs[8].resocounter += oscs[8].resofreq;
    outvalL += (tab_sine[oscs[8].resocounter>>7] * (65535-oscs[8].counter)>>9) * mod_resonance.value;
  }

  oscs[9].counter += oscs[9].freq;
  index = oscs[9].counter + (oscs[9].modulator[oscs[9].counter>>6] * oscs[9].mod_depth);
  result = oscs[9].waveform[index>>6];
  //env
  if(oscs[9].result_prev < 0 && result > 0)
    oscs[9].env_val = oscs[9].envelope;
  oscs[9].result_prev = result;
  outvalL += result * oscs[9].env_val;
  //reso
  if(oscs[9].counter < 32768)
    oscs[9].resocounter = 0;
  else
  {
    oscs[9].resocounter += oscs[9].resofreq;
    outvalL += (tab_sine[oscs[9].resocounter>>7] * (65535-oscs[9].counter)>>9) * mod_resonance.value;
  }
  
  //do some FX =================================================================  
  outvalR = outvalL;

  if(chor_active)
  {
    //chorus left channel +++++++++++++++++++++++++++++++++++++++++++++++++
    chor_bufferL[chor_counter] = outvalL;
   
    chor_counter++;
    if(chor_counter >= chor_depth) 
    {
      chor_counter = 0; 
      if(chor_count_up)
      {
         chor_bufferL[chor_depth] = chor_bufferL[chor_depth - 1]; 
         chor_bufferL[chor_depth + 1] = chor_bufferL[chor_depth - 1]; 
         
         chor_depth++;
         if (chor_depth >= chor_max)
          chor_count_up = false;
      }
      else
      {
         chor_depth--;
         if (chor_depth <= CHOR_MIN_D)
          chor_count_up = true;
      }
    }
   
    outvalL += chor_bufferL[chor_counter];
    outvalL >> 1; //double the audio so half the volume
  
    //chorus right channel +++++++++++++++++++++++++++++++++++++++++++++++++
    chor_bufferR[chor_counterR] = outvalR;
   
    chor_counterR++;
    if(chor_counterR >= chor_depthR) 
    {
      chor_counterR = 0; 
      if(chor_count_upR)
      {
         chor_bufferR[chor_depthR] = chor_bufferR[chor_depthR - 1]; 
         chor_bufferR[chor_depthR + 1] = chor_bufferR[chor_depthR - 1]; 
         
         chor_depthR++;
         if (chor_depthR >= chor_max)
          chor_count_upR = false;
      }
      else
      {
         chor_depthR--;
         if (chor_depthR <= CHOR_MIN_D)
          chor_count_upR = true;
      }
    }
   
    outvalR += chor_bufferR[chor_counterR];
    outvalR >> 1;
  }
  
  
  //convert to unsigned 4096 for output
  int16_t outputL = outvalL >> 13;
  outputL += 2048;

  int16_t outputR = outvalR >> 13;
  outputR += 2048;
  
  //write to DAC's =============================================================
  dacc_set_channel_selection(DACC_INTERFACE, 0);       //select DAC channel 0
  dacc_write_conversion_data(DACC_INTERFACE, outputL); //write to DAC

  dacc_set_channel_selection(DACC_INTERFACE, 1);       //select DAC channel 0
  dacc_write_conversion_data(DACC_INTERFACE, outputR); //write to DAC

  //increment timing counter
  timercounter++;
}


