/*************************************************************
* This file contains all the voice related vars and funcs
* for the Electone HS organ.
*
* by Jeroen van Kuik, 2023
*************************************************************/

//vars =================================================
//current voice registration sysex code
//tells the HS which registration is being edited
uint8_t voice_reg = 0x08;

//array of all param values (77 bytes)
//these are used for the currently edited voice
//and is initialized with values for a new voice
uint8_t voice_params[77] =
{
  //VOICE:
  0b00000000,  //0 = orig_voice: original voice number? not sure
  0b00000000,  //1 = portamento level
  0b00000000,  //2 = portamento time
  0b00000000,  //3 = upper shift, pedal shift, DFL filter enable
  0b00000000,  //4 = Symphonic Enable, Symphonic mode: Celeste or Symphonic, Tone grp. enable, Tone grp. select
  0b00000000,  //5 = aftertouch depth
  0b00000000,  //6 = osc sync, feedback, algorithm

  //OPERATORS:
  0b00000001,  //7 =  0, op4 freq fixed, op4 freq coarse
  0b00000001,  //8 =  0, op2 freq fixed, op2 freq coarse
  0b00000001,  //9 =  0, op3 freq fixed, op3 freq coarse
  0b00000001,  //10 = 0, op1 freq fixed, op1 freq coarse
  0b10000000,  //11 = 1, op4 waveform, op4 freq fine
  0b10000000,  //12 = 1, op2 waveform, op2 freq fine
  0b10000000,  //13 = 1, op3 waveform, op2 freq fine
  0b10000000,  //14 = 1, op1 waveform, op1 freq fine
  0b01100000,  //15 = op4 sustain enable, op4 level
  0b01100000,  //16 = op2 sustain enable, op2 level
  0b01100000,  //17 = op3 sustain enable, op3 level
  0b00000011,  //18 = op1 sustain enable, op1 level
  0b00011111,  //19 = op4 KS rate, op4 ratio or fixed, op4 attack
  0b00011111,  //20 = op2 KS rate, op2 ratio or fixed, op2 attack
  0b00011111,  //21 = op3 KS rate, op3 ratio or fixed, op3 attack
  0b00011111,  //22 = op1 KS rate, op1 ratio or fixed, op1 attack
  0b00011111,  //23 = op4 AMS enable, op4 AMS depth, op4 decay1
  0b00011111,  //24 = op2 AMS enable, op2 AMS depth, op2 decay1
  0b00011111,  //25 = op3 AMS enable, op3 AMS depth, op3 decay1
  0b00011111,  //26 = op1 AMS enable, op1 AMS depth, op1 decay1
  0b00000000,  //27 = op4 detune, hardcoded 0, op4 decay2
  0b00000000,  //28 = op2 detune, hardcoded 0, op2 decay2
  0b00000000,  //29 = op3 detune, hardcoded 0, op3 decay2
  0b00000000,  //30 = op1 detune, hardcoded 0, op1 decay2
  0b00001111,  //31 = op4 decay level, op4 release
  0b00001111,  //32 = op2 decay level, op2 release
  0b00001111,  //33 = op3 decay level, op3 release
  0b0001111,  //34 = op1 decay level, op1 release
  0b10000011,  //35 = op4 KS left, op4 KS level, op4 KS right
  0b10000011,  //36 = op2 KS left, op2 KS level, op2 KS right
  0b10000011,  //37 = op3 KS left, op3 KS level, op3 KS right
  0b10000011,  //38 = op1 KS left, op1 KS level, op1 KS right
  0b00100000,  //39 = op4 EG shift, hardcoded 1, op4 velocity AR, op4 velocity sens
  0b00100000,  //40 = op2 EG shift, hardcoded 1, op2 velocity AR, op2 velocity sens
  0b00100000,  //41 = op3 EG shift, hardcoded 1, op3 velocity AR, op3 velocity sens
  0b00100000,  //42 = op1 EG shift, hardcoded 1, op1 velocity AR, op1 velocity sens
  0b00000000,  //43 = op4 harmonic, op4 harmonic level
  0b00000000,  //44 = op2 harmonic, op2 harmonic level
  0b00000000,  //45 = op3 harmonic, op3 harmonic level
  0b00000000,  //46 = op1 harmonic, op1 harmonic level

  //LFO:       //LFO1 is used for the Lead Attack and Amplitude Modulation
               //LFO2 is used for Lead and Orchestra Pitch Modulation (vibrato)
  0b00000000,  //47 = LFO1 lead attack PMS, LFO1 level, LFO1 AMS
  0b00000000,  //48 = LFO1 rate
  0b00000000,  //49 = LFO1 PMD (1) or AMD (0) mode, LFO1 PMD or AMD depth
  0b00000000,  //50 = LFO1 PMS, LFO1 waveform
  0b00000000,  //51 = LFO1 attack time
  0b00000000,  //52 = LFO2 enable?, LFO2 PMS, ??
  0b00000000,  //53 = LFO2 rate
  0b00000000,  //54 = LFO2 PMD depth
  0b00000000,  //55 = LFO2 waveform
  0b00000000,  //56 = LFO2 lead delay time

  //DFL:       //Special filter IC YM2406, Orchestra and Combi sounds get filtered through this
               //these bytes are all zeroes for all other sounds (and don't get send)
  0b00000000,  //57 = DFL filter gain, DFL filter offset
  0b00000000,  //58 = DFL filter t0
  0b00000000,  //59 = DFL filter t1
  0b00000000,  //60 = DFL filter k1
  0b00000000,  //61 = DFL filter t2
  0b00000000,  //62 = DFL filter k2
  0b00000000,  //63 = DFL filter t3
  0b00000000,  //64 = DFL filter k3
  0b00000000,  //65 = DFL filter t4
  0b00000000,  //66 = DFL filter k4
  0b00000000,  //67 = DFL filter t5
  0b00000000,  //68 = DFL filter k5
  0b00000000,  //69 = DFL filter t6
  0b00000000,  //70 = DFL filter k6
  0b00000000,  //71 = DFL filter t7
  0b00000000,  //72 = DFL filter k7
  0b00000000,  //73 = DFL filter t8
  0b00000000,  //74 = DFL filter k8
  0b00000000,  //75 = DFL S or D, DFL input shift
  0b00000000  //76 = (DFL) Filter enable
};

const PROGMEM uint8_t voice_max[20][5] =
{
//VOICE
  1, 1, 15, 255, 99,  //gen A (keyboard) upper shift, pedal shift, aftertouch, porta lvl, porta time
  7, 7, 1, 1, 1,      //gen B (voice)    algo, feedback (op4), osc sync, AMD/PMD sw, DFL preset (0 = off i.e. DFL enable)
  255, 127, 7, 3, 3,  //gen C (LFO1)     rate, AM depth, PMS, AMS, waveform
  255, 127, 3, 255, 3,  //gen D (LFO2)   rate, ?? depth, PMS, Lead attack, waveform 

//OPERATOR 1
  99, 7, 3, 1, 7, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  15, 15, 3, 7, 1, //op B (frequency) coarse, fine, detune, fix, mode
  31, 31, 15, 31, 15, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  1, 3, 15, 7, 3, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right

//OPERATOR 2
  99, 7, 3, 1, 7, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  15, 15, 3, 7, 1, //op B (frequency) coarse, fine, detune, fix, mode
  31, 31, 15, 31, 15, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  1, 3, 15, 7, 3, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right

//OPERATOR 3
  99, 7, 3, 1, 7, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  15, 15, 3, 7, 1, //op B (frequency) coarse, fine, detune, fix, mode
  31, 31, 15, 31, 15, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  1, 3, 15, 7, 3, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right

//OPERATOR 4
  99, 7, 3, 1, 7, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  15, 15, 3, 7, 1, //op B (frequency) coarse, fine, detune, fix, mode
  31, 31, 15, 31, 15, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  1, 3, 15, 7, 3, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right
};

const PROGMEM uint8_t voice_indexes[20][5] = 
{
//VOICE
  3, 3, 5, 0, 1,      //gen A (keyboard) upper shift, pedal shift, aftertouch, porta lvl, porta time
  6, 6, 6, 49, 3,     //gen B (voice)    algo, feedback (op4), osc sync, AMD/PMD sw, DFL preset (0 = off i.e. DFL enable)
  48, 49, 47, 47, 50, //gen C (LFO1) rate, depth, PMS, AMS, waveform
  53, 54, 52, 51, 55, //gen D (LFO2)     rate, depth, PMS, Lead attack, waveform 

//OPERATOR 1
  18, 42, 42, 26, 14, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  10, 14, 30, 10, 22, //op B (frequency) coarse, fine, detune, fix, mode
  22, 26, 34, 30, 34, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  18, 42, 38, 38, 38, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right

//OPERATOR 2
  16, 40, 40, 24, 12, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  8, 12, 28, 8, 20, //op B (frequency)   coarse, fine, detune, fix, mode
  20, 24, 32, 28, 32, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  16, 40, 36, 36, 36, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right

//OPERATOR 3
  17, 41, 41, 25, 13, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  9, 13, 29, 9, 21, //op B (frequency)   coarse, fine, detune, fix, mode
  21, 25, 33, 29, 33, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  17, 41, 37, 37, 37, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right

//OPERATOR 4
  15, 39, 39, 23, 11, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  7, 11, 27, 7, 19, //op B (frequency)   coarse, fine, detune, fix, mode
  19, 23, 31, 27, 31, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  15, 39, 35, 35, 35  //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right
};

const PROGMEM uint8_t voice_bitmasks[20][5] = 
{
//VOICE
  0b10000000, 0b01000000, 0b11110000, 0b11111111, 0b11111111, //gen A (keyboard) upper shift, pedal shift, aftertouch, porta lvl, porta time
  0b00000111, 0b00111000, 0b11000000, 0b10000000, 0b00100000, //gen B (voice)    algo, feedback (op4), osc sync, AMD/PMD sw, DFL preset (0 = off i.e. DFL enable)
  0b11111111, 0b01111111, 0b01110000, 0b00000011, 0b00000011, //gen C (LFO1)     rate, depth, PMS, AMS, waveform
  0b11111111, 0b11111111, 0b00110000, 0b11111111, 0b00000011, //gen D (LFO2)     rate, depth, PMS, Lead attack, waveform 

//OPERATOR 1
  0b01111111, 0b00000111, 0b00011000, 0b10000000, 0b01110000, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  0b00001111, 0b00001111, 0b11000000, 0b01110000, 0b00100000, //op B (frequency) coarse, fine, detune, fix, mode
  0b00011111, 0b00011111, 0b11110000, 0b00011111, 0b00001111, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  0b10000000, 0b11000000, 0b01111100, 0b10000000, 0b00000011, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right

//OPERATOR 2
  0b01111111, 0b00000111, 0b00011000, 0b10000000, 0b01110000, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  0b00001111, 0b00001111, 0b11000000, 0b01110000, 0b00100000, //op B (frequency) coarse, fine, detune, fix, mode
  0b00011111, 0b00011111, 0b11110000, 0b00011111, 0b00001111, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  0b10000000, 0b11000000, 0b01111100, 0b10000000, 0b00000011, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right

//OPERATOR 3
  0b01111111, 0b00000111, 0b00011000, 0b10000000, 0b01110000, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  0b00001111, 0b00001111, 0b11000000, 0b01110000, 0b00100000, //op B (frequency) coarse, fine, detune, fix, mode
  0b00011111, 0b00011111, 0b11110000, 0b00011111, 0b00001111, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  0b10000000, 0b11000000, 0b01111100, 0b10000000, 0b00000011, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right

//OPERATOR 4
  0b01111111, 0b00000111, 0b00011000, 0b10000000, 0b01110000, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  0b00001111, 0b00001111, 0b11000000, 0b01110000, 0b00100000, //op B (frequency) coarse, fine, detune, fix, mode
  0b00011111, 0b00011111, 0b11110000, 0b00011111, 0b00001111, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  0b10000000, 0b11000000, 0b01111100, 0b10000000, 0b00000011, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right
};

const PROGMEM uint8_t voice_bitshifts[20][5] = 
{
//VOICE
  7, 6, 4, 0, 0, //gen A (keyboard) upper shift, pedal shift, aftertouch, porta lvl, porta time
  0, 3, 6, 7, 5, //gen B (voice)    algo, feedback (op4), osc sync, AMD/PMD sw, DFL preset (0 = off i.e. DFL enable)
  0, 0, 4, 0, 0, //gen C (LFO1)     rate, AM depth, PMS, AMS, waveform
  0, 0, 4, 0, 0, //gen D (LFO2)     rate, ?? depth, PMS, Lead attack, waveform 

//OPERATOR 1
  0, 0, 3, 7, 4, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  0, 0, 6, 4, 5, //op B (frequency) coarse, fine, detune, fix, mode
  0, 0, 4, 0, 0, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  7, 6, 2, 7, 0, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right

//OPERATOR 2
  0, 0, 3, 7, 4, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  0, 0, 6, 4, 5, //op B (frequency) coarse, fine, detune, fix, mode
  0, 0, 4, 0, 0, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  7, 6, 2, 7, 0, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right

//OPERATOR 3
  0, 0, 3, 7, 4, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  0, 0, 6, 4, 5, //op B (frequency) coarse, fine, detune, fix, mode
  0, 0, 4, 0, 0, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  7, 6, 2, 7, 0, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right

//OPERATOR 4
  0, 0, 3, 7, 4, //op A (amp)       level, velocity, vel. AR, AMS enable, waveform
  0, 0, 6, 4, 5, //op B (frequency) coarse, fine, detune, fix, mode
  0, 0, 4, 0, 0, //op C (envelope)  attack, dec1, dec lvl, dec2, release 
  7, 6, 2, 7, 0, //op D (misc)      sustain enable, EG shift, KS level, KS left, KS right
};


//funcs ================================================
//Sends a single parameter update to the HS
void sendParam(uint8_t index, uint8_t value)
{
  //calculate checksum
  uint8_t checksum = value & 0b01111111;
  
  //tell organ to start receiving a parameter update
  //0x43, 0x70, 0x70, 0x02, ID1, ID2, adress low, adress high, count low, count high
  uint8_t message_command[10] = {0x43, 0x70, 0x72, 0x02, voice_reg, 0x00, index, 0x00, 0x01, 0x00};
  MIDI.sendSysEx(10, message_command);

  //convert and send the parameter data
	//0x43, 0x70, 0x72, 0x00, data, checksum
  if(value > 0b00111111)
  {
    uint8_t data[2];
    data[0] = (value & 0b00111111) | 0b01000000;
    data[1] = ((value & 0b11000000) >> 2) | 0b01000000;

    uint8_t message_data[7] = {0x43, 0x70, 0x72, 0x00, data[0], data[1], checksum};
    MIDI.sendSysEx(7, message_data);
  }
  else
  {
    uint8_t message_data[6] = {0x43, 0x70, 0x72, 0x00, value, checksum};
    MIDI.sendSysEx(6, message_data);
  }
}


//sends an entire voice to the HS
void sendVoice()
{
  uint8_t datasize = 77;
  if (voice_reg == 0x08 || voice_reg == 0x10)
    datasize = 77;
  else
    datasize = 57;

  //could do sendSB command but with adress starting at zero and byte count at 77
  //0x43, 0x70, 0x70, 0x02, ID1, ID2, adress low, adress high, count low, count high
  uint8_t message_command[10] = {0x43, 0x70, 0x72, 0x02, voice_reg, 0x00, 0x00, 0x00, datasize, 0x00};
  MIDI.sendSysEx(10, message_command);

  //now generate checksum from the array
  uint8_t checksum = 0;
  for(uint8_t i = 0; i < datasize; i++)
    checksum += voice_params[i];
  checksum &= 0b01111111;

  //reserve a bunch of memory which should be enough for
  //the extra bytes from the conversion + the sysex bytes
  //{0x43, 0x70, 0x72, 0x00, values[], checksum};
  uint8_t message_data[datasize * 2];
  message_data[0] = 0x43;
  message_data[1] = 0x70;
  message_data[2] = 0x72;
  message_data[3] = 0x00;

  uint8_t data_index = 4;
  for(uint8_t i = 0; i < datasize; i++)
  {
    uint8_t value = voice_params[i];
    if(value > 0b00111111)
    {
      message_data[data_index] = (value & 0b00111111) | 0b01000000;
      message_data[data_index + 1] = ((value & 0b11000000) >> 2) | 0b01000000;
      data_index += 2;
    }
    else
    {
      message_data[data_index] = value;
      data_index++;
    }
  }
  message_data[data_index] = checksum;
  data_index++;

  //send converted data, I can cheat here since you can give an array size smaller than the
  //actual array you're passing onto this method
  MIDI.sendSysEx(data_index, message_data);
}