/*************************************************************
* Main project file, primarily contains the user interface
* vars and funcs.
*
* This project is still very much a work-in-progress
* but should provide a solid basis to iterate upon.
*
* by Jeroen van Kuik, 2023
*************************************************************/

//libs =================================================
#include <EEPROM.h> 
#include <MIDI.h>
#include <Wire.h>
#include <LiquidCrystal_I2C.h>

LiquidCrystal_I2C lcd(0x27, 16, 2);
MIDI_CREATE_DEFAULT_INSTANCE();

//includes =============================================
#include "hs_voice.h"

//defines ==============================================
#define slider_param A0 //parameter adjust slider
#define slider_X A1     //parameter select slider
#define slider_Y A2     //menu select slider

#define btn_cancel 2    //receive/cancel button (red)
#define btn_confirm 3   //send/confirm button (green)
#define btn_op1 4       //operator/page select button 1
#define btn_op2 5       //operator/page select button 2
#define btn_op3 6       //operator/page select button 3
#define btn_op4 7       //operator/page select button 4

#define menu_storage 0  //internal storage menu
#define menu_gen_A 1
#define menu_gen_B 2
#define menu_gen_C 3
#define menu_gen_D 4
#define menu_op_A 5
#define menu_op_B 6
#define menu_op_C 7
#define menu_op_D 8
#define menu_send 9     //send/receive menu
#define menu_save 10    //save voice menu
#define menu_load 11    //load voice menu

//vars =================================================
//timing
uint16_t timer_param = 0;

//menu navigation
uint8_t menu_current = 0;
uint8_t menu_prev = 255;
uint8_t param_current = 0;
uint8_t param_prev = 255;
uint8_t param_value = 0;
uint8_t param_value_prev = 0;
uint16_t btn_cancel_debounce = 0;
uint16_t btn_confirm_debounce = 0;

uint8_t op_current = 0;
uint8_t op_addr[4] = {3, 1, 2, 0}; //because operators are arranged 4,2,3,1 yay...

uint8_t storage_slot = 0;
uint8_t storage_page = 0;

//current voice name
char voice_name[4] = {'M','P','T','Y'};


//funcs ================================================
//displays the current menu on the LCD
//sets the blinking cursor to the current param
void menu_set()
{
  lcd.clear();

  switch(menu_current)
  {
    case menu_storage:
    {
      lcd.setCursor(0,0);
      lcd.print("STORAGE PAGE: " + String(storage_page + 1));
      lcd.setCursor(0,1);
      uint8_t addr = (storage_page * 12);
      char voin[4];
      voin[0] = EEPROM.read(addr);
      voin[1] = EEPROM.read(addr + 1);
      voin[2] = EEPROM.read(addr + 2);
      voin[3] = EEPROM.read(addr + 3);
      lcd.print(voin);
      lcd.setCursor(6,1);
      addr += 4;
      voin[0] = EEPROM.read(addr);
      voin[1] = EEPROM.read(addr + 1);
      voin[2] = EEPROM.read(addr + 2);
      voin[3] = EEPROM.read(addr + 3);
      lcd.print(voin);
      lcd.setCursor(12,1);
      addr += 4;
      voin[0] = EEPROM.read(addr);
      voin[1] = EEPROM.read(addr + 1);
      voin[2] = EEPROM.read(addr + 2);
      voin[3] = EEPROM.read(addr + 3);
      lcd.print(voin);

      //blinkie
      lcd.setCursor((storage_slot * 6), 1);
      break;
    }
    case menu_gen_A: //gen A (keyboard)  upper shift, pedal shift, aftertouch, porta lvl, porta time
    {
      lcd.setCursor(2,0);
      lcd.print("US");
      lcd.setCursor(5,0);
      lcd.print("PS");
      lcd.setCursor(8,0);
      lcd.print("AF");
      lcd.setCursor(11,0);
      lcd.print("PL");
      lcd.setCursor(14,0);
      lcd.print("PT");
      
      lcd.setCursor(2,1);
      lcd.print((voice_params[3] & 0b10000000) >> 7); //upper shift
      lcd.setCursor(5,1);
      lcd.print((voice_params[3] & 0b01000000) >> 6); //pedal shift
      lcd.setCursor(8,1);
      lcd.print((voice_params[5] & 0b11110000) >> 4); //aftertouch
      lcd.setCursor(11,1);
      lcd.print(voice_params[0]); //porta lvl
      lcd.setCursor(14,1);
      lcd.print(voice_params[1]); //porta time

      //blinkie
      lcd.setCursor((param_current * 3) + 3, 1);
      break;
    }
    case menu_gen_B: //gen B (voice)   algo, feedback (op4), osc sync, AMD/PMD sw, DFL preset (0 = off i.e. DFL enable)
    {
      lcd.setCursor(2,0);
      lcd.print("AL");
      lcd.setCursor(5,0);
      lcd.print("FB");
      lcd.setCursor(8,0);
      lcd.print("SY");
      lcd.setCursor(11,0);
      lcd.print("LF");
      lcd.setCursor(14,0); //TODO presets
      lcd.print("DF");
      
      lcd.setCursor(2,1);
      lcd.print(voice_params[6] & 0b00000111); //algo
      lcd.setCursor(5,1);
      lcd.print((voice_params[6] & 0b00111000) >> 3); //feedback
      lcd.setCursor(8,1);
      lcd.print((voice_params[6] & 0b11000000) >> 6); //sync
      lcd.setCursor(11,1);
      lcd.print(((voice_params[49] & 0b10000000) >> 7) + 1); //AMD/PMD sw i.e. LFO 1 or 2 select
      lcd.setCursor(14,1);
      lcd.print((voice_params[3] & 0b00100000) >> 5); //TODO: DFL presets ... hmmmm

      //blinkie
      lcd.setCursor((param_current * 3) + 3, 1);
      break;
    }
    case menu_gen_C: //gen C (LFO1)    rate, depth, PMS, AMS, waveform
    {
      lcd.setCursor(2,0);
      lcd.print("RT");
      lcd.setCursor(5,0);
      lcd.print("DE");
      lcd.setCursor(8,0);
      lcd.print("PD");
      lcd.setCursor(11,0);
      lcd.print("AD");
      lcd.setCursor(14,0);
      lcd.print("WA");
      
      lcd.setCursor(2,1);
      lcd.print(voice_params[48]); //LFO1 rate
      lcd.setCursor(5,1);
      lcd.print(voice_params[49] & 0b01111111); //depth
      lcd.setCursor(8,1);
      lcd.print((voice_params[47] & 0b01110000) >> 4); //PMS depth
      lcd.setCursor(11,1);
      lcd.print(voice_params[47] & 0b00000011); //AMS depth
      lcd.setCursor(14,1);
      lcd.print(voice_params[50] & 0b00000011); //waveform

      //blinkie
      lcd.setCursor((param_current * 3) + 3, 1);
      break;
    }
    case menu_gen_D: //gen D (LFO2)    rate, depth, PMS, Lead attack, waveform 
    {
      lcd.setCursor(2,0);
      lcd.print("RT");
      lcd.setCursor(5,0);
      lcd.print("DE");
      lcd.setCursor(8,0);
      lcd.print("PD");
      lcd.setCursor(11,0);
      lcd.print("LA");
      lcd.setCursor(14,0);
      lcd.print("WA");
      
      lcd.setCursor(2,1);
      lcd.print(voice_params[53]); //LFO2 rate
      lcd.setCursor(5,1);
      lcd.print(voice_params[54]); //depth
      lcd.setCursor(8,1);
      lcd.print((voice_params[52] & 0b00110000) >> 4); //PMS depth
      lcd.setCursor(11,1);
      lcd.print(voice_params[51]); //lead attack time
      lcd.setCursor(14,1);
      lcd.print(voice_params[55] & 0b00000011); //waveform

      //blinkie
      lcd.setCursor((param_current * 3) + 3, 1);
      break;
    }
    case menu_op_A: //op A (amp)    level, velocity, vel. AR, AMS depth, waveform
    {
      lcd.setCursor(0,0);
      lcd.print("O");
      lcd.setCursor(0,1);
      lcd.print(String(op_current + 1));

      lcd.setCursor(2,0);
      lcd.print("LV");
      lcd.setCursor(5,0);
      lcd.print("VE");
      lcd.setCursor(8,0);
      lcd.print("AR");
      lcd.setCursor(11,0);
      lcd.print("AM");
      lcd.setCursor(14,0);
      lcd.print("WA");
      
      lcd.setCursor(2,1);
      lcd.print(100 - (voice_params[15 + op_addr[op_current]] & 0b01111111)); //level
      lcd.setCursor(5,1);
      lcd.print(voice_params[39 + op_addr[op_current]] & 0b00000111); //velocity
      lcd.setCursor(8,1);
      lcd.print((voice_params[39 + op_addr[op_current]] & 0b00011000) >> 3); //vel AR
      lcd.setCursor(11,1);
      lcd.print((voice_params[23 + op_addr[op_current]] & 0b10000000) >> 7); //AMS enable
      lcd.setCursor(14,1);
      lcd.print((voice_params[11 + op_addr[op_current]] & 0b01110000) >> 4); //waveform

      //blinkie
      lcd.setCursor((param_current * 3) + 3, 1);
      break;
    }
    case menu_op_B: //op B (frequency) coarse, fine, detune, fix, mode
    {
      lcd.setCursor(0,0);
      lcd.print("O");
      lcd.setCursor(0,1);
      lcd.print(String(op_current + 1));

      lcd.setCursor(2,0);
      lcd.print("FC");
      lcd.setCursor(5,0);
      lcd.print("FF");
      lcd.setCursor(8,0);
      lcd.print("DE");
      lcd.setCursor(11,0);
      lcd.print("FI");
      lcd.setCursor(14,0);
      lcd.print("MO");
      
      lcd.setCursor(2,1);
      lcd.print(voice_params[7 + op_addr[op_current]] & 0b00001111); //coarse
      lcd.setCursor(5,1);
      lcd.print(voice_params[11 + op_addr[op_current]] & 0b00001111); //fine
      lcd.setCursor(8,1);
      lcd.print((voice_params[27 + op_addr[op_current]] & 0b11000000) >> 6); //detune
      lcd.setCursor(11,1);
      lcd.print((voice_params[7 + op_addr[op_current]] & 0b01110000) >> 4); //fix freq
      lcd.setCursor(14,1);
      lcd.print((voice_params[19 + op_addr[op_current]] & 0b00100000) >> 5); //ratio/fix mode

      //blinkie
      lcd.setCursor((param_current * 3) + 3, 1);
      break;
    }
    case menu_op_C: //op C (envelope)  attack, dec1, dec lvl, dec2, release 
    {
      lcd.setCursor(0,0);
      lcd.print("O");
      lcd.setCursor(0,1);
      lcd.print(String(op_current + 1));

      lcd.setCursor(2,0);
      lcd.print("AR");
      lcd.setCursor(5,0);
      lcd.print("D1");
      lcd.setCursor(8,0);
      lcd.print("DL");
      lcd.setCursor(11,0);
      lcd.print("D2");
      lcd.setCursor(14,0);
      lcd.print("RR");
      
      lcd.setCursor(2,1);
      lcd.print(voice_params[19 + op_addr[op_current]] & 0b00011111); //A
      lcd.setCursor(5,1);
      lcd.print(voice_params[23 + op_addr[op_current]] & 0b00011111); //D1
      lcd.setCursor(8,1);
      lcd.print(15 - ((voice_params[31 + op_addr[op_current]] & 0b11110000) >> 4)); //DL
      lcd.setCursor(11,1);
      lcd.print(voice_params[27 + op_addr[op_current]] & 0b00011111); //D2
      lcd.setCursor(14,1);
      lcd.print(voice_params[31 + op_addr[op_current]] & 0b00001111); //R

      //blinkie
      lcd.setCursor((param_current * 3) + 3, 1);
      break;
    }
    case menu_op_D: //op D (misc)      sustain enable, EG shift, KS level, KS rate, KS type
    {
      lcd.setCursor(0,0);
      lcd.print("O");
      lcd.setCursor(0,1);
      lcd.print(String(op_current + 1));

      lcd.setCursor(2,0);
      lcd.print("SU");
      lcd.setCursor(5,0);
      lcd.print("EG");
      lcd.setCursor(8,0);
      lcd.print("KL");
      lcd.setCursor(11,0);
      lcd.print("K<");
      lcd.setCursor(14,0);
      lcd.print("K>");
      
      lcd.setCursor(2,1);
      lcd.print((voice_params[15 + op_addr[op_current]] & 0b10000000) >> 7); //sustain enable
      lcd.setCursor(5,1);
      lcd.print((voice_params[39 + op_addr[op_current]] & 0b11000000) >> 6); //EG shift
      lcd.setCursor(8,1);
      lcd.print((voice_params[35 + op_addr[op_current]] & 0b01111100) >> 2); //KS level
      lcd.setCursor(11,1);
      lcd.print((voice_params[35 + op_addr[op_current]] & 0b10000000) >> 7); //KS left
      lcd.setCursor(14,1);
      lcd.print(voice_params[35 + op_addr[op_current]] & 0b00000011); //KS mid + right

      //blinkie
      lcd.setCursor((param_current * 3) + 3, 1);
      break;
    }
    case menu_send:
    {
      lcd.setCursor(0,0);
      lcd.print("SEND VOICE TO:");

      lcd.setCursor(0,1);
      lcd.print("U.Or");
      //lcd.setCursor(5,1);
      //lcd.print("EG");
      lcd.setCursor(6,1);
      lcd.print("L.Or");
      //lcd.setCursor(11,1);
      //lcd.print("KR");
      lcd.setCursor(12,1);
      lcd.print("Lead");

      //blinkie
      lcd.setCursor(storage_slot * 6, 1);
      break;
    }
    case menu_save:
    {
      lcd.setCursor(0,0);
      lcd.print("SET VOICE NAME:");
      lcd.setCursor(0, 1);
      lcd.print(voice_name);
      lcd.setCursor(0, 1);
      break;
    }
    case menu_load:
    {
      char v_name[4];
      uint8_t addr = (storage_page * 12) + (storage_slot * 4);
      v_name[0] = EEPROM.read(addr);
      v_name[1] = EEPROM.read(addr + 1);
      v_name[2] = EEPROM.read(addr + 2);
      v_name[3] = EEPROM.read(addr + 3);

      lcd.setCursor(0,0);
      lcd.print("Load: " + String(v_name) + "?");
      break;
    }
  }
}


void setup() 
{
  //init display
  Wire.begin();
  lcd.init();
  lcd.clear();         
  lcd.backlight();
  lcd.blink();

  //init MIDI
  MIDI.begin();

  //prevent insta-send
  param_value = analogRead(slider_param) >> 3;
  param_value_prev = param_value;
}


void loop() 
{
  //read and select menu
  param_current = analogRead(slider_X) >> 6;
  if (param_current <= 2)
  {
    param_current = 0;
    if(menu_current == menu_storage || menu_current == menu_send)
      storage_slot = 0;
  }
  else if(param_current > 2 && param_current < 6)
  {
    param_current = 1;
    if(menu_current == menu_storage || menu_current == menu_send)
      storage_slot = 0;
  }
  else if(param_current >= 6 && param_current < 11)
  {
    param_current = 2;
    if(menu_current == menu_storage || menu_current == menu_send)
      storage_slot = 1;
  }
  else if(param_current >= 11 && param_current < 15)
  {
    param_current = 3;
    if(menu_current == menu_storage || menu_current == menu_send)
      storage_slot = 1;
  }
  else
  {
    param_current = 4;
    if(menu_current == menu_storage || menu_current == menu_send)
      storage_slot = 2;
  }
  
  if(param_current != param_prev)
  {
    param_prev = param_current;
    menu_set();
  }

  if(menu_current < menu_send)
  {
    menu_current = analogRead(slider_Y) >> 4;
    if(menu_current > 0 && menu_current < 5)
      menu_current = 1;
    else if(menu_current >= 5 && menu_current < 14)
      menu_current = 2;
    else if(menu_current >= 14 && menu_current < 24)
      menu_current = 3;
    else if(menu_current >= 24 && menu_current < 34)
      menu_current = 4;
    else if(menu_current >= 34 && menu_current < 45)
      menu_current = 5;
    else if(menu_current >= 45 && menu_current < 56)
      menu_current = 6;
    else if(menu_current >= 56 && menu_current < 63)
      menu_current = 7;
    else if(menu_current == 63)
      menu_current = 8;
  
    if(menu_current != menu_prev)
    {
      menu_prev = menu_current;
      menu_set();
    }
  }
  
  //param values when editing a voice
  if(menu_current != 0 && menu_current < menu_send)
  {
    //don't constantly send updates, otherwise
    //the midi buffer might get overwhelmed
    timer_param++;
    if(timer_param == 1500)
    {
      timer_param = 0;
      //configure current parameter
      param_value = analogRead(slider_param) >> 3;
      if(param_value != param_value_prev)
      {
        param_value_prev = param_value;

        //TODO:
        //gotta figure out how to adjust the params
        //gotta use a bitmask and bitshift
        //could do voice_params[i] &= ~bitmask; to clear the old value
        //then voice_params[i] |= (param_value << bitshift) & bitmask; to add the new value

        //I guess I could make 4 arrays, with the param max, param index, bitmask & shift info
        //so these should be in PROGMEM, since they are read-only
        uint8_t index = menu_current -1;
        if(menu_current >= menu_op_A)
          index += op_current * 4;

        uint8_t v_index = pgm_read_byte(&voice_indexes[index][param_current]);
        uint8_t v_max = pgm_read_byte(&voice_max[index][param_current]);
        uint8_t v_mask = pgm_read_byte(&voice_bitmasks[index][param_current]);
        uint8_t v_shift = pgm_read_byte(&voice_bitshifts[index][param_current]);
        
        uint8_t param_final = 0;
        if((index == 4 || index == 8 || index == 12 || index == 16) && param_current == 0) //inv level
          param_final = map(param_value, 0, 127, v_max, 0);
        else if((index == 6 || index == 10 || index == 14 || index == 18) && param_current == 2) //inv dec level
          param_final = map(param_value, 0, 127, v_max, 0);
        else
          param_final = map(param_value, 0, 127, 0, v_max);

        //set new value bits
        voice_params[v_index] &= ~v_mask;
        voice_params[v_index] |= (param_final << v_shift) & v_mask;

        //bit 7 is hardcoded to 0 on these bytes:
        if(v_index >= 7 && v_index <= 10)
          voice_params[v_index] &= 0b01111111;
        //bit 7 is hardcoded to 1 on these bytes:
        if(v_index >= 11 && v_index <= 14)
          voice_params[v_index] |= 0b10000000;

        //set entire LCD screen, doing just the param is too much of a PITA
        menu_set();

        //send param change to organ
        sendParam(v_index, voice_params[v_index]);
      }
    }
  }
  //voice name when saving
  else if(menu_current == menu_save)
  {
    uint8_t index = min(param_current, 3);
    char letter = map(analogRead(slider_param), 0, 1023, 65, 90);
    if(letter != voice_name[index])
    {
      voice_name[index] = letter;
      lcd.setCursor(index, 1);
      lcd.print(voice_name[index]);
      lcd.setCursor(index, 1);
    }
  }

  //operator buttons
  //in voice edit mode these will select the current operator being edited
  //in storage mode these select the current page
  if(menu_current > menu_gen_D && menu_current < menu_send)
  {
    if(digitalRead(btn_op1) == HIGH)
    {
      op_current = 0;
      menu_set();
    }
    else if(digitalRead(btn_op2) == HIGH)
    {
      op_current = 1;
      menu_set();
    }
    else if(digitalRead(btn_op3) == HIGH)
    {
      op_current = 2;
      menu_set();
    }
    else if(digitalRead(btn_op4) == HIGH)
    {
      op_current = 3;
      menu_set();
    }
  }
  else if(menu_current == 0)
  {
    if(digitalRead(btn_op1) == HIGH)
    {
      storage_page = 0;
      menu_set();
    }
    else if(digitalRead(btn_op2) == HIGH)
    {
      storage_page = 1;
      menu_set();
    }
    else if(digitalRead(btn_op3) == HIGH)
    {
      storage_page = 2;
      menu_set();
    }
    else if(digitalRead(btn_op4) == HIGH)
    {
      storage_page = 3;
      menu_set();
    }
  }

  //confirm/cancel buttons
  //these buttons will also act as send/receive buttons when in voice edit mode (menu_current != 0)
  //a menu should appear which selects which registration the voice will be send to/received from
  //when in storage mode these buttons are to confirm or cancel saving and setting voice names
  //these do need to be debounced!
  if(digitalRead(btn_confirm) == HIGH && btn_confirm_debounce == 0)
  {
    btn_confirm_debounce = 900;

    //when in storage mode, show save menu
    if(menu_current == 0)
    {
      menu_prev = menu_current;
      menu_current = menu_save;
      menu_set();
    }
    //when editing a voice: show send menu
    else if(menu_current > 0 && menu_current < menu_send)
    {
      menu_prev = menu_current;
      menu_current = menu_send;
      menu_set();
    }
    //when sending/receiving a voice
    else if(menu_current == menu_send)
    {
      //for now just set the voice_reg:
      if(param_current == 0) //upper orch
        voice_reg = 0x08;
      else if(param_current == 2) //lower orch
        voice_reg = 0x10;
      else if(param_current == 4) //lead
        voice_reg = 0x00;

      lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print("Sending Voice...");

      sendVoice();

      //give an happy message before returning
      lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print("Success!");
      delay(1000);
      
      //finally go back to previous menu
      menu_current = menu_prev;
      menu_set();
    }
    //when editing a voice name
    else if(menu_current == menu_save)
    {
      //EEPROM ARRANGEMENT (4 pages of 3 slots):
      //0 - 99: voice names
      //100 - 1023 voice data

      lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print("Storing...");

      //store name to EEPROM
      //name sections start at byte 0
      uint8_t addr = (storage_page * 12) + (storage_slot * 4);
      EEPROM.update(addr, voice_name[0]);
      EEPROM.update(addr + 1, voice_name[1]);
      EEPROM.update(addr + 2, voice_name[2]);
      EEPROM.update(addr + 3, voice_name[3]);

      //dump 77 byte array to EEPROM
      //data sections start at byte 100
      addr = (storage_page * 231) + (storage_slot * 77);
      for(uint8_t i = 0; i < 77; i++)
        EEPROM.update(100 + addr + i, voice_params[i]);

      //finally go back to previous menu
      menu_current = menu_prev;
      menu_set();
    }
    else if(menu_current == menu_load)
    {
      //load voice data from EEPROM
      lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print("Loading...");

      //name sections start at byte 0
      uint8_t addr = (storage_page * 12) + (storage_slot * 4);
      voice_name[0] = EEPROM.read(addr);
      voice_name[1] = EEPROM.read(addr + 1);
      voice_name[2] = EEPROM.read(addr + 2);
      voice_name[3] = EEPROM.read(addr + 3);

      //get 77 byte array from EEPROM
      //data sections start at byte 100
      addr = (storage_page * 231) + (storage_slot * 77);
      for(uint8_t i = 0; i < 77; i++)
        voice_params[i] = EEPROM.read(100 + addr + i);
      
      //finally go back to previous menu
      menu_current = menu_prev;
      menu_set();
    }

  }
  if(digitalRead(btn_cancel) == HIGH && btn_cancel_debounce == 0)
  {
    btn_cancel_debounce = 900;

    //when in storage mode, show load menu
    if(menu_current == 0)
    {
      menu_prev = menu_current;
      menu_current = menu_load;
      menu_set();
    }
    else if(menu_current == menu_save)
    {
      menu_current = menu_prev;
      menu_set();
    }
    else if(menu_current == menu_load)
    {
      menu_current = menu_prev;
      menu_set();
    }
    else if(menu_current == menu_send)
    {
      menu_current = menu_prev;
      menu_set();
    }
  }

  //process the debouncing
  //note I'm not using a timer, since the loop is 
  //already slow enough for the counts to take time
  if(btn_confirm_debounce > 0)
    btn_confirm_debounce--;
  if(btn_cancel_debounce > 0)
    btn_cancel_debounce--;
}
