//POLY800 CONTROL PROCESSOR

//Port manipulation macros
#define CLR(x,y)(x&=(~(1<<y)))
#define SET(x,y)(x|=(1<<y))

#define DELAY_CYCLES(n) __builtin_avr_delay_cycles(n)

//PORTA
#define FP_JOY_Y A0
#define FP_SPEED A1
#define CHOR_EN 2
#define CHOR_SPD 3
#define CHOR_DPT 4
#define LE_KB_SW 5
#define OE_KB 7
#define OE_SW 6

//PORTB
#define LE_LA 0
#define LE_LB 1
#define LE_DAC 2
#define SH_INH 3

//PORTD
#define DCO_WR 2
#define DCO_REG1 3
#define DCO_REG2 4

//7 SEGMENT DISPLAY CHARACTERS
#define DISP_0 0b01110111
#define DISP_1 0b01000100
#define DISP_2 0b01011011
#define DISP_3 0b01011110
#define DISP_4 0b01101100
#define DISP_5 0b00111110
#define DISP_6 0b00111111
#define DISP_7 0b01010100
#define DISP_8 0b01111111
#define DISP_9 0b01111110

#define DISP_DOT 0b10000000
#define DISP_LINE 0b00001000

#define DISP_A 0b01111101
#define DISP_d 0b01001111
#define DISP_E 0b00111011
#define DISP_G 0b00110111
#define DISP_L 0b00100011
#define DISP_o 0b00001111
#define DISP_P 0b01111001
#define DISP_T 0b01010100
#define DISP_Y 0b01101100

//FRONT PANEL KEYCODES
#define KEY_STEP 0
#define KEY_START 1
#define KEY_POLY 2    //sequencer record
#define KEY_CHORD 13  //sequencer clear
#define KEY_HOLD 14   //something else I guess? maybe it -is- gonna be hold
#define KEY_DOWN 11
#define KEY_UP 12
#define KEY_WRITE 9

#define KEY_BANKHOLD 15
#define KEY_PROG 10
#define KEY_1 23
#define KEY_2 22
#define KEY_3 21
#define KEY_4 20
#define KEY_5 19
#define KEY_6 18
#define KEY_7 17
#define KEY_8 16

//S&H adresses
#define SH_DCO1_LVL 0
#define SH_DCO2_LVL 1
#define SH_RESO 2
#define SH_CUTOFF 3
#define SH_NOISE 4
#define SH_VIBO 5

//INCLUDES
#include "presets.h"
#include "tables.h"

//GLOBALS
volatile byte timer_disp = 0;
volatile byte timer_keyscan = 0;
volatile byte timer_lfo = 0;
volatile byte timer_slowstuff = 0;
bool ADC_stage = false;

//MIDI
byte midi_state = 0;
byte midi_note = 0;
byte midi_vel = 0;
bool midi_gate = false;

//LFO/MG
volatile int16_t MG_phase = 0;
int16_t MG_delay = 0;
volatile bool MG_stage = false;

//Display
byte disp_segsel = 1;
byte disp_datasel = 0;
byte disp_data[6] = {0};
byte disp_conv[10] = 
{
  DISP_0,
  DISP_1,
  DISP_2,
  DISP_3,
  DISP_4,
  DISP_5,
  DISP_6,
  DISP_7,
  DISP_8,
  DISP_9,
};

//keyboard
bool gate = false;
bool singtrig = true;
bool keystates[49];
byte buttstates[24];
byte currbutt = 0;

//menus
byte menumode = 0;
byte currpreset = 11;
byte currparam = 11;
byte* parampt = NULL;
bool keypad_digit = false;
byte keypad_number = 0;

//ADBSSR envelope
int16_t env_val = 0;
byte env_stage = 0;
byte env_noise = 0;
byte env_filter = 0;

//sequencer
byte seq_notes[256];
byte seq_index = 0;
byte seq_max = 0;
byte seq_stoi = 0;
byte seq_state = 0;
uint16_t seq_tempo = 3906;

// FUNCTIONS ==============================================================================
void DispClear()
{
  disp_data[0] = 0;
  disp_data[1] = 0;
  disp_data[2] = 0;
  disp_data[3] = 0;
  disp_data[4] = 0;
  disp_data[5] = 0;
}

void DispVal(byte value, byte disp)
{
  byte sega = value / 10;
  byte segb = value - (sega * 10);  

  byte dispsel = disp * 2;
  disp_data[dispsel] = disp_conv[sega];
  disp_data[dispsel+1] = disp_conv[segb];
}

void setup() 
{
  //configure ports
  DDRA |= 0b11111100; //keys, buttons control and ADC
  PORTA |= 0b11000100;
  DDRB |= 0b00001111; //7-seg, DAC control and ISP
  PORTB = 0b00001000;
  DDRC = 0b11111111; //databus
  PORTC = 0b00000000;
  DDRD |= 0b10011100; //DCO control and MIDI/serial
  PORTD = 0b00000000;

  //2 analog inputs, joystick Y and sequencer speed slider
  //so need to set the ADC
  ADCSRA = (1<<ADEN) | (1<<ADPS2) | (1<<ADPS0); //turn on , div of 32 so 250KHz
  ADMUX |= (1<<REFS0); //internal AVCC with external capacitor
  //request first sample
  ADMUX = 0b01000000 | 0b00000001; //pin 1 (seq speed)
  ADCSRA |= 0b01000000; //set ADSC bit to request new conversion

  //clear all latches (databus is already set to 0)
  SET(PORTA, LE_KB_SW);
  CLR(PORTA, LE_KB_SW);
  SET(PORTB, LE_LA);
  CLR(PORTB, LE_LA);
  SET(PORTB, LE_LB);
  CLR(PORTB, LE_LB);
  SET(PORTB, LE_DAC);
  CLR(PORTB, LE_DAC);

  //display test
  PORTC = 0b11111111;
  SET(PORTB, LE_LA);
  CLR(PORTB, LE_LA);
  SET(PORTB, LE_LB);
  CLR(PORTB, LE_LB);
  delay(1000);
  PORTC = 0b00000000;
  SET(PORTB, LE_LA);
  CLR(PORTB, LE_LA);
  SET(PORTB, LE_LB);
  CLR(PORTB, LE_LB);

  //ensure sequencer is initialized
  for(byte i = 0; i != 255; i++)
  {
    seq_notes[i] = 0b01000000;
  }
  
  cli();
  //timer 0 for the 4Khz interrupt
  TCCR0 = 0;
  TCNT0 = 0;
  OCR0 = 31; //~4kHz
  TCCR0 |= (1 << WGM01);   //CTC mode
  TCCR0 |= 0b00000011;    //prescaler: 64
  TIMSK |= (1 << OCIE0);  //enable timer compare interrupt

  //timer 1 for our dear sequencer lmao why do I exist
  TCCR1A = 0;
  TCCR1B = 0;
  TCNT1  = 0;
  TCCR1B |= (1 << CS12);   //256 prescaler = 31250
  //60 BPM, 16th note  = 250ms / 4Hz
  //120 BPM, 16th note = 125ms / 8Hz
  //250 BPM, 16th note = 60ms  / ~17Hz    
  sei();

  Serial.begin(31250);

  //set display to default
  DispVal(currpreset, 0);
  disp_data[3] = DISP_P;

  //load first preset
  LoadPreset(11);

  if(PARAMS[7])
    disp_data[2] = DISP_DOT;
}

//main loop, fetches MIDI, scans keys/buttons, updates front panel display, processes and sends data over databus/PORTC
void loop() 
{
  if(Serial.available())
  {
    byte comm = Serial.read();

    //1 status byte:
    //note off = 1000nnnn (n = channel)
    //note on =  1001nnnn (n = channel)
    //2 bytes after that: pitch and velocity
    //if velocity is zero it counts as a note off event

    //some controllers do not resent the status byte if it hasn't changed
    //some will also never send a "note off" message, but rather a "note on with zero velocity"
    //need to take this into consideration!
    if(comm > 127) //message
    {
      comm &= 0b11110000; //remove channel, will implement that part later
      if(comm == 0b10010000) //note on
        midi_state = 1;
      else if(comm == 0b10000000) //note off
        midi_state = 2;
      else if(comm == 0b10110000) //message
        midi_state = 4;
    }
    else //data
    {
      if(midi_state == 1) //note on note
      {
        if(comm > 23 and comm < 84)
        {
          midi_note = comm - 24;
          midi_state = 3;
        }
      }
      else if(midi_state == 2) //note off note
      {
        if(comm > 23 and comm < 84)
        {
          //send NOTE to DCO  
          DDRC = 0b11111111; //PORTC as output for note data
          PORTD |= 0b00011000; //note data    
          PORTC = 0b10000000; //Note OFF          
          PORTC += comm - 24;
          SET(PORTD, DCO_WR);
          //nees to be set for at least 36us to ensure fetch
          DELAY_CYCLES(400);
          CLR(PORTD, DCO_WR);
          
          midi_gate = false;
        }
        midi_state = 0;
      }
      else if(midi_state == 3) //note on velocity
      {
        midi_vel = comm;
        //send NOTE to DCO  
        DDRC = 0b11111111; //PORTC as output for note data
        PORTD |= 0b00011000; //note data    

        if(midi_vel)
        {
          PORTC = 0b11000000; //Note ON
          midi_gate = true;
        }
        else
        {
          PORTC = 0b10000000; //Note OFF if velocity zero
          midi_gate = false;
        }
        
        PORTC += midi_note;
        SET(PORTD, DCO_WR);
        //nees to be set for at least 36us to ensure fetch
        DELAY_CYCLES(400);
        CLR(PORTD, DCO_WR);

        midi_vel = 0;
        midi_note = 0;
        midi_state = 0;
      }
      else if(midi_state == 4 and comm == 123) //all note off message
      {
        SendDCOData(27, 2); //command, NOTE PANIC 
        midi_gate = false;
        midi_state = 0;
      }
    }
    
  }

  //sequencer
  if(seq_state == 2) //play mode
  {
    if(TCNT1 > seq_tempo) //when having reached slider value
    {
      TCNT1 = 0;

      DispVal(seq_index, 2);
      
      //also need to send note off of previous note(s)...
      while(seq_stoi) //there are notes previously played!
      {
        //send NOTE OFF to DCO
        DDRC = 0b11111111; //PORTC as output for note data
        PORTD |= 0b00011000; //note data    
        PORTC = 0b10000000; //Note OFF
        PORTC += seq_notes[seq_index - seq_stoi] & 0b00111111;
        SET(PORTD, DCO_WR);
        DELAY_CYCLES(400);
        CLR(PORTD, DCO_WR);
  
        seq_stoi--;
      }

      if(seq_index > seq_max)
        seq_index = 0;

      //check if this is a rest, just increment
      if(seq_notes[seq_index] & 0b01000000)
      {
        seq_index++;
        midi_gate = false;
      }
      else //if not, play notes in this step
      {
        while(1)
        {
          //send NOTE ON to DCO  
          DDRC = 0b11111111; //PORTC as output for note data
          PORTD |= 0b00011000; //note data          
          PORTC = 0b11000000; //Note ON
          PORTC += seq_notes[seq_index] & 0b00111111;
          SET(PORTD, DCO_WR);
          DELAY_CYCLES(400);
          CLR(PORTD, DCO_WR);
    
          //step stop note indexer and sequencer and stop if end of sequence is reached
          //breaking here will also prevent user forgetting to press "step" button at end of sequence
          //causing mayhem (I hope)

          if(seq_notes[seq_index] & 0b10000000)
            break;
          seq_stoi++;
          seq_index++;
          if(seq_index > seq_max)
            break;
          
        }
        midi_gate = true;
        seq_index++;
        seq_stoi++;
      }
    }
  }  

  //========================================================================================
  //process ENV =================================================================
  if(timer_lfo > 15) //~250Hz
  {
    timer_lfo = 0;

    //envelope: ADBSSR (DEG3)
    if(gate or midi_gate)
    {
      if(env_stage == 0) //attack
      {
        env_val += EXPCONV[PARAMS[48]];
        if(env_val > 4095)
        {
          env_val = 4095;
          env_stage = 1;
        }
      }
      else if(env_stage == 1) //decay (to breakpoint)
      {
        env_val -= EXPCONV[PARAMS[49]];
        if(env_val < LINCONV[PARAMS[50]])
        {
          env_val = LINCONV[PARAMS[50]];
          env_stage = 2;
        }
      }
      else if(env_stage == 2) //slope (to sustain)
      {
        if(PARAMS[50] < PARAMS[52])
        {
          env_val += EXPCONV[PARAMS[51]];
          if(env_val > LINCONV[PARAMS[52]])
          {
            env_val = LINCONV[PARAMS[52]];
            env_stage = 3;
          }
        }
        else
        {
          env_val -= EXPCONV[PARAMS[51]];
          if(env_val < LINCONV[PARAMS[52]])
          {
            env_val = LINCONV[PARAMS[52]];
            env_stage = 3;
          }
        }
      }      
    }
    else //release
    {      
      env_stage = 0;
      if(env_val > EXPCONV[PARAMS[53]])
      {
        env_val -= EXPCONV[PARAMS[53]];
      }
      else
      {
        env_val = 0;
      }        
    }

    //TODO: keyboard tracking, param 26
    //TODO: joystick VCF depth
    
    if(PARAMS[27])
      env_filter = min((PARAMS[24]*8) + (((env_val >> 1) * PARAMS[28])/128), 255);
    else
      env_filter = max((PARAMS[24]*8) - (((env_val >> 1) * PARAMS[28])/128), 0);

    env_noise = ((uint16_t)env_val * PARAMS[18]) >> 8;
  }

  //========================================================================================
  //update display and S&H =================================================================
  if(timer_disp > 7) //~500Hz
  {
    timer_disp = 0;

    PORTC = 0b00000000; //clear port
    DDRC = 0b11111111; //PORTC as output
    PORTC = disp_data[disp_datasel];
    SET(PORTB, LE_LA);
    CLR(PORTB, LE_LA);

    PORTC = disp_segsel;
    SET(PORTB, LE_LB);
    CLR(PORTB, LE_LB);

    disp_datasel++;
    disp_segsel *= 2;
    if(disp_segsel == 64)
    {
      disp_segsel = 1;
      disp_datasel = 0;
    }

    //UPDATE SAMPLE AND HOLD
    PORTC = PARAMS[6] * 8;
    SET(PORTB, LE_DAC);
    CLR(PORTB, LE_DAC);    
    PORTC = SH_DCO1_LVL;
    CLR(PORTB, SH_INH);
    DELAY_CYCLES(500);
    SET(PORTB, SH_INH);  

    PORTC = PARAMS[14] * 8;
    SET(PORTB, LE_DAC);
    CLR(PORTB, LE_DAC);    
    PORTC = SH_DCO2_LVL;
    CLR(PORTB, SH_INH);
    DELAY_CYCLES(500);
    SET(PORTB, SH_INH);  

    PORTC = env_filter;
    SET(PORTB, LE_DAC);
    CLR(PORTB, LE_DAC);
    PORTC = SH_CUTOFF;
    CLR(PORTB, SH_INH);
    DELAY_CYCLES(500);
    SET(PORTB, SH_INH);  

    PORTC = PARAMS[25] * 16;
    SET(PORTB, LE_DAC);
    CLR(PORTB, LE_DAC);    
    PORTC = SH_RESO;
    CLR(PORTB, SH_INH);
    DELAY_CYCLES(500);
    SET(PORTB, SH_INH);  

    PORTC = env_noise;
    SET(PORTB, LE_DAC);
    CLR(PORTB, LE_DAC);    
    PORTC = SH_NOISE;
    CLR(PORTB, SH_INH);
    DELAY_CYCLES(500);
    SET(PORTB, SH_INH);  

    //TODO: joystick DCO depth
    PORTC = (((MG_phase * (int16_t)PARAMS[58])/512)*MG_delay)/200 + 134; //value needs to be centered at close to 2.5V
    SET(PORTB, LE_DAC);
    CLR(PORTB, LE_DAC);    
    PORTC = SH_VIBO;
    CLR(PORTB, SH_INH);
    DELAY_CYCLES(500);
    SET(PORTB, SH_INH);     
  }

  //========================================================================================
  //scan buttons and keyboard ==============================================================
  if(timer_keyscan > 7) //~500Hz
  {
    timer_keyscan = 0;

    byte keys_column = 1;
    PORTC = 0b00000000; //clear port
    DDRC = 0b11111111; //PORTC as output
    PORTC = 0b00000000;
    SET(PORTA, LE_KB_SW);
    CLR(PORTA, LE_KB_SW);

    gate = false;

    for(byte col = 0; col < 8; col++)
    {
      //set driver latch
      PORTC = 0b00000000; //clear port
      DDRC = 0b11111111; //PORTC as output
      PORTC = keys_column;
      SET(PORTA, LE_KB_SW);
      CLR(PORTA, LE_KB_SW);

      //======================================
      //fetch front panel button presses
      PORTC = 0b00000000; //clear port
      DDRC = 0b00000000; //PORTC as input
      CLR(PORTA, OE_SW); //active low!
      DELAY_CYCLES(5);
      byte data = PINC & 0b00000111; //FP button data is now on bus/PORTC     
      
      byte rind = 1;
      for(byte row = 0; row < 3; row++)
      {    
        byte buttind = (row*8) + col;
        if(data & rind) //this button is pressed
        {
          if(!buttstates[buttind])
          {
            buttstates[buttind] = 64;
            currbutt = buttind;      
          }
        }
        else //this button is not pressed
        {
          if(buttstates[buttind])
          {
            buttstates[buttind]--;
          }
        }

        rind *= 2;
      }
      
      SET(PORTA, OE_SW); //turn off latch output again

      //======================================
      //fetch keyboard presses
      CLR(PORTA, OE_KB); //active low!
      DELAY_CYCLES(5);
      data = PINC; //keyboard data is now on bus/PORTC      
      
      //polyphony requires to process every row
      rind = 128;
      SET(PORTA, OE_KB); //turn off latch output again    

      DDRC = 0b11111111; //PORTC as output for note data

      for(byte row = 6; row > 0; row--)
      {
        byte ind = 55 - ((row*8) + col);
        if(data & rind) //keypress
        {
          gate = true;
          
          if(keystates[ind] == false)
          {
            keystates[ind] = true;

            //trigger
            if(PARAMS[29])
              env_stage = 0;
            else if(singtrig)
            {
              env_stage = 0;
              singtrig = false;
            }
              
            //send NOTE ON to DCO  
            PORTD |= 0b00011000; //note data          
            PORTC = 0b11000000; //Note ON
            PORTC += ind;
            SET(PORTD, DCO_WR);
            //nees to be set for at least 36us to ensure fetch
            DELAY_CYCLES(400);
            CLR(PORTD, DCO_WR);

            //sequencer
            if(seq_state == 1 and seq_index < 250)
            {
              seq_notes[seq_index] |= ind; //set note
              seq_notes[seq_index] &= 0b10111111; //clear rest bit
              seq_index++;
              seq_max = seq_index;
              DispVal(seq_index, 2);
            }
          }
        }
        else //not pressed
        {
          if(keystates[ind] == true)
          {
            keystates[ind] = false;
            //send NOTE OFF to DCO
            PORTD |= 0b00011000; //note data    
            PORTC = 0b10000000; //Note OFF
            PORTC += ind;
            SET(PORTD, DCO_WR);
            DELAY_CYCLES(400);
            CLR(PORTD, DCO_WR);
          }
        }
        //if(row == 0)
        //  break;
        rind /= 2;
      }  
      keys_column *= 2;
    } 

    if(!gate)
      singtrig = true;

    DDRC = 0b11111111; //PORTC as output
    PORTC = 0b00000000; //clear
  }

  //========================================================================================
  //process data, menus etc, all the non time critical stuff goes here =====================
  if(timer_slowstuff > 133) //~30Hz
  {
    timer_slowstuff = 0;

    //ADC
    if(!(ADCSRA & 0b01000000)) //conversion complete, ADSC is clear
    {
      uint16_t adc_val = ADCW; //fetch value
      
      if(ADC_stage) //joystick Y
      {
        //TODO: figure out center value of joystick
        //everything above this value is DCO vibo
        //everything under this value is VCF mod
        
        
        //next sample
        ADMUX = 0b01000000 | 0b00000001; //pin 1 (seq speed)
        ADCSRA |= 0b01000000; //set ADSC bit to request new conversion
      }
      else //seq speed
      {
        seq_tempo = adc_val * 10;
        
        //next sample
        ADMUX = 0b01000000;// | 0b00000000 = pin 0 (joystick)
        ADCSRA |= 0b01000000; //set ADSC bit to request new conversion
      }
      
      ADC_stage = !ADC_stage;
    }
    
    //MG delay
    if(gate and MG_delay < 200)
      MG_delay += 16-PARAMS[57];
    else if(!gate and MG_delay > 0)
      MG_delay--;

    //menus
    if(currbutt != 255)
    {    
      if(menumode == 1 and parampt != NULL) 
      {
        byte paramindex = ParamToIndex(currparam);
        bool change = false;
        if(currbutt == KEY_UP and *parampt < PARAM_LIMITS[paramindex])
        {
          (*parampt)++;
          change = true;
        }
        else if(currbutt == KEY_DOWN and *parampt > 0)
        {
          (*parampt)--;
          change = true;
        }        

        //will need to update data and settings in the case of some params
        //all these if statements could be replaced by a switch tbh
        if(change)
        {
          if(paramindex == 31) //chorus
          {
            if(PARAMS[31])
              CLR(PORTA, CHOR_EN);
            else
              SET(PORTA, CHOR_EN);

            //TODO: once wired up
            //set speed and depth
          }          
          
          //DCO 1
          if(paramindex == 0)
            SendDCOData(paramindex, PARAMS[0]);
            
          if(paramindex > 0 and paramindex < 6)
          {
            SendDCOData(paramindex, PARAMS[paramindex]);
            SendDCOData(27, 0); //command, GENTABLES A
            DELAY_CYCLES(8000);
          }

          //DCO 2
          if(paramindex == 8)
            SendDCOData(6, PARAMS[8]);
            
          if(paramindex > 8 and paramindex < 14)
          {
            SendDCOData(paramindex-2, PARAMS[paramindex]);
            SendDCOData(27, 1); //command, GENTABLES B
            DELAY_CYCLES(8000);
          }

          if(paramindex == 16)
            SendDCOData(12, PARAMS[16]);
          if(paramindex == 17)
            SendDCOData(13, PARAMS[17]);

          //ENV 1
          if(paramindex > 31 and paramindex < 38)
            SendDCOData(paramindex-18, PARAMS[paramindex]);

          //ENV 2
          if(paramindex > 39 and paramindex < 46)
            SendDCOData(paramindex-20, PARAMS[paramindex]);
        }
        
        DispVal(*parampt, 2);
      }

      //sequencer
      if(currbutt == KEY_POLY) //seq record mode
      {
        if(seq_state == 1) //stop recording
        {
          seq_state = 0;
          DispClear();
          menumode = 0;          
          DispVal(currpreset, 0);
          disp_data[3] = DISP_P;
          if(PARAMS[7])
            disp_data[2] = DISP_DOT;
        }
        else
        {
          //unload current param to prevent beepletisms
          parampt = NULL;
          menumode = 3; //ensure the other menus are disabled
            
          seq_state = 1;    
  
          DispClear();
          disp_data[1] = DISP_5;
          DispVal(seq_index, 2);
        }
      }
      else if(currbutt == KEY_START) //seq play mode
      {
        if(seq_state == 2) //stop
        {
          SendDCOData(27, 2); //command, NOTE PANIC
                    
          midi_gate = false;
          menumode = 0;
          seq_state = 0;
          seq_index = seq_max; //to allow continued programming
          seq_stoi = 0;

          disp_data[4] = 0b00000000;
          disp_data[5] = 0b00000000;
        }
        else //play
        {
          //reset everything, EVERYTHING
          menumode = 3;
          seq_state = 2;
          seq_index = 0;
          seq_stoi = 0;
          TCNT1 = 0; //reset timer
          TIFR &= 0b11101111; //set timer flag (by setting to 0), this should immediately start the first notes

          disp_data[4] = DISP_P;
          disp_data[5] = DISP_5;
        }
      }
      else if(currbutt == KEY_CHORD and seq_state == 1) //clear seq
      {
        seq_index = 0;
        seq_max = 0;
        seq_stoi = 0;
        
        for(byte i = 0; i != 255; i++)
        {          
          seq_notes[i] = 0b01000000;
        }

        DispClear();
        disp_data[1] = DISP_5;
        DispVal(seq_index, 2);
      }      
      else if(currbutt == KEY_STEP and seq_index < 250) //seq step
      {
        if(seq_notes[seq_index - 1] & 0b10000000) //previous step bit is set
        {
          //increment
          seq_index++;
          seq_max = seq_index;    
          DispVal(seq_index, 2);  
        }
        else
        {        
          //set prev entry timing bit to indicate end of that step
          seq_notes[seq_index - 1] |= 0b10000000;
        }          
      }
                    
      //function keys
      if(currbutt == KEY_PROG)
      {
        DispClear();
        if(menumode == 0) //go into programming mode
        {
          menumode = 1;
          DispVal(currpreset, 0);

          //reset param to ensure no fuckery
          currparam = 11;
          parampt = &PARAMS[ParamToIndex(currparam)];
          DispVal(currparam, 1);
          DispVal(*parampt, 2);          
        }
        else if (menumode == 1) //go into preset select mode
        {
          menumode = 0;

          //unload current param
          parampt = NULL;
          
          DispVal(currpreset, 0);
          disp_data[3] = DISP_P;
          if(PARAMS[7])
            disp_data[2] = DISP_DOT;
        }
      }    

      //number keypad
      if(currbutt > 15 and currbutt < 24) //a keypad key is pressed
      {
        if(keypad_digit)
        {
          keypad_number += (24 - currbutt);
          //when fully entered apply this number to anything active
          if(menumode == 0) //select program
          {
            currpreset = keypad_number;
            DispVal(currpreset, 0);
            
            if(currpreset < 50)
            {
              LoadPreset(currpreset);              
            }
            else //the rest is user presets on EEPROM
            {
              LoadPresetEEPROM(currpreset);
            }            
          }
          else if(menumode == 1) //select a param
          {
            currparam = keypad_number;
            DispVal(currparam, 1);
            parampt = &PARAMS[ParamToIndex(currparam)];  
            DispVal(*parampt, 2);          
          }   

          keypad_number = 0;
          keypad_digit = false;
        }
        else
        {
          keypad_number += (24 - currbutt)*10;          
          keypad_digit = true;

          if(menumode == 0)
          {
            DispVal(keypad_number, 0);
            disp_data[1] = DISP_LINE;
          }
          else if(menumode == 1)
          {
            DispVal(keypad_number, 1);
            disp_data[3] = DISP_LINE;
          }
        }
      }
      
      currbutt = 255;
    }
  }
}

//4KHz interrupt, updates timers, LFO phase and other time-critical stuffs
ISR(TIMER0_COMP_vect)
{
  //update LFO and such
  if(MG_stage)
  {
    MG_phase -= PARAMS[56];
    if(MG_phase < -1500)
      MG_stage = false;
  }
  else
  {
    MG_phase += PARAMS[56];
    if(MG_phase > 1500)
      MG_stage = true;
  }
  
  //update timers
  timer_disp++;
  timer_keyscan++;
  timer_lfo++;
  timer_slowstuff++;
}
